<?php
/**
 * @package        Alter Fields
 * @copyright      Copyright (C) 2022-2023 AlterBrains.com. All rights reserved.
 * @license        https://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL
 */

/** @noinspection PhpMultipleClassDeclarationsInspection */

namespace AlterBrains\Plugin\System\Alterfields\Extension;

use Joomla\CMS\Application\AdministratorApplication;
use Joomla\CMS\Application\SiteApplication;
use Joomla\CMS\Document\HtmlDocument;
use Joomla\CMS\Event\CustomFields\AfterPrepareFieldEvent;
use Joomla\CMS\Event\CustomFields\BeforePrepareFieldEvent;
use Joomla\CMS\Event\CustomFields\PrepareFieldEvent;
use Joomla\CMS\Event\Model;
use Joomla\CMS\Factory;
use Joomla\CMS\Fields\FieldsServiceInterface;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Layout\FileLayout;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Profiler\Profiler;
use Joomla\CMS\Uri\Uri;
use Joomla\Component\Fields\Administrator\Helper\FieldsHelper;
use Joomla\Component\Fields\Administrator\Model\FieldsModel;
use Joomla\Database\DatabaseAwareInterface;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Event\SubscriberInterface;
use Joomla\Filesystem\Path;
use Joomla\Registry\Registry;

\defined('_JEXEC') or die;

/**
 * @since        1.0
 * @noinspection PhpUnused
 */
class Alterfields extends CMSPlugin implements SubscriberInterface, DatabaseAwareInterface
{
    use DatabaseAwareTrait;

    /**
     * @var SiteApplication|AdministratorApplication
     * @since        1.0
     * @noinspection PhpMissingFieldTypeInspection
     */
    private $application;

    /**
     * @since 1.0
     */
    protected static ?array $context = null;

    /**
     * @since 1.0
     */
    protected static array $contextMap;

    /**
     * @var \stdClass[]
     * @since 1.0
     */
    protected static ?array $fields = null;

    /**
     * All fields, required to get subform children which can be missed in $fields.
     * @var \stdClass[]
     * @since 1.2.0
     */
    protected static ?array $fieldsAll = null;

    /**
     * @since 1.0
     */
    public static ?Registry $listModelState = null;

    /**
     * @inheritDoc
     * @since 1.0
     */
    public function __construct(array $config = [])
    {
        parent::__construct($config);

        $this->application = Factory::getApplication();

        self::$contextMap = (array)$this->params->get('contextMap', []);
    }

    /**
     * @inheritDoc
     * @since 1.0
     */
    public static function getSubscribedEvents(): array
    {
        return [
            'onContentPrepareForm' => 'onContentPrepareForm',
            'onAfterDispatch'      => 'onAfterDispatch',
        ];
    }

    /** @since  1.0 */
    public function onContentPrepareForm(Model\PrepareFormEvent $event): void
    {
        $form = $event->getForm();
        $data = $event->getData();

        switch ($form->getName()) {
            case 'com_fields.field.' . $this->application->getInput()->get('context'):
                if (!empty($data->type)) {
                    $this->loadLanguage();
                    $form->loadFile(__DIR__ . '/../../forms/com_fields.field.xml', false);
                    // todo, 'repeatable' is Joomla3 EOL
                    if ($data->type === 'subform' || $data->type === 'repeatable') {
                        $form->removeField('af_orderable', 'params');
                        if (!empty($data->params['af_orderable'])) {
                            $data->params['af_orderable'] = '0';
                        }
                    }
                }
                break;

            case 'com_fields.fields.filter':
                if ($this->params->get('backend_fields_info')) {
                    $this->decorateFieldList();
                }
                break;
        }
    }

    /** @since 1.0 */
    public function onAfterDispatch(): void
    {
        JDEBUG and Profiler::getInstance('Application')->mark('before Alter Fields list');

        if ($context = static::getContext()) {
            $this->decorateList($context);
        }

        JDEBUG and Profiler::getInstance('Application')->mark('after Alter Fields list');
    }

    /** @since 1.0 */
    protected function decorateList(array $context): void
    {
        // Get fields to display in columns
        $fields = static::getFields(static function ($field) {
            return $field->af_listable && empty($field->only_use_in_subform);
        }, $context);
        if (!$fields) {
            return;
        }

        if (\function_exists('ini_set') && ($limit = $this->params->get('pcre_backtrack_limit'))) {
            \ini_set('pcre_backtrack_limit', $limit);
        }

        /** @var HtmlDocument $document */
        $document = $this->application->getDocument();

        $document->setBuffer(
            \preg_replace_callback(
            //'~<table(.+?)name="checkall-toggle"(.+?)</table>~s',
                '~<table\b[^>]*>.*?name="checkall-toggle".*?</table>~s',
                function ($matches) use ($context, $fields) {
                    return $this->decorateListTable($context, $matches[0], $fields);
                },
                $document->getBuffer('component') ?? '',
                1
            ),
            'component'
        );

        // Show PCRE error
        if (($document->getBuffer('component') ?? '') === '' && (0 !== $errorCode = \preg_last_error())) {
            $errorText = $errorCode === \PREG_BACKTRACK_LIMIT_ERROR
                ? 'PLG_SYSTEM_ALTERFIELDS_ERROR_PCRE_BACKTRACK_LIMIT_EXHAUSTED'
                : 'PLG_SYSTEM_ALTERFIELDS_ERROR_PCRE';
            $this->loadLanguage();
            $this->application->enqueueMessage(
                Text::sprintf($errorText, \preg_last_error_msg(), \ini_get('pcre.backtrack_limit')),
                'error'
            );
        }
    }

    /**
     * @param \stdClass[] $fields
     *
     * @since 1.0
     */
    protected function decorateListTable(array $context, string $html, array $fields): string
    {
        // Extract thead.
        $html = \preg_replace_callback(
        //'~<thead(.+?)</thead>~us',
            '~<thead\b[^>]*>.*?</thead>~s',
            function ($matches) use ($context, $fields) {
                return $this->decorateListTableHead($context, $matches[0], $fields);
            },
            $html,
            1
        );

        // Extract tbody.
        return \preg_replace_callback(
        //'~<tbody(.+?)</tbody>~us',
            '~<tbody\b[^>]*>.*?</tbody>~s',
            function ($matches) use ($context, $fields) {
                return $this->decorateListTableBody($context, $matches[0], $fields);
            },
            $html,
            1
        );
    }

    /**
     * @param \stdClass[] $fields
     *
     * @since 1.0
     * @noinspection PhpUnusedParameterInspection
     */
    protected function decorateListTableHead(array $context, string $html, array $fields): string
    {
        if (self::$listModelState) {
            $listOrder = \htmlspecialchars(self::$listModelState->get('list.ordering', ''));
            $listDirn = \htmlspecialchars(self::$listModelState->get('list.direction', ''));
        } else {
            $listOrder = '';
            $listDirn = '';
        }

        $columns = [];
        $append = [];

        foreach ($fields as $field) {
            if ($field->af_orderable) {
                // Field key in request.
                $field->af_name = 'af_' . $field->id;

                $title = HTMLHelper::_(
                    'searchtools.sort',
                    $field->af_list_title,
                    $field->af_name,
                    $listDirn,
                    $listOrder
                );
            } else {
                $title = \htmlspecialchars($field->af_list_title);
            }

            $class = $field->params->get('af_th_class');
            $fieldHtml = '<th' . ($class ? ' class="' . $class . '"' : '') . '>' . $title . ' </th>';

            if ($field->af_column) {
                $columns[$field->af_column] = $fieldHtml;
            } else {
                $append[] = $fieldHtml;
            }
        }

        if ($columns) {
            $col = 0;

            $html = \preg_replace_callback(
                '~>\s*?((<t[hd])|(</tr))~u',
                static function ($place) use (&$col, &$columns) {
                    while (isset($columns[++$col])) {
                        $new ??= '';
                        $new .= $columns[$col];
                        // just for memory
                        unset($columns[$col]);
                    }

                    return isset($new) ? '>' . $new . $place[1] : $place[0];
                },
                $html
            );
        }

        return $append ? \strtr($html, ['</tr>' => \implode("\n", $append) . '</tr>']) : $html;
    }

    /**
     * @param \stdClass[] $fields
     *
     * @since 1.0
     */
    protected function decorateListTableBody(array $context, string $html, array $fields): string
    {
        $tbody = $html;

        // Values
        //\preg_match_all('~<td(.+?)name="cid\[]" value="(\d+)"(.+?)</tr>~s', $tbody, $matches);
        //\preg_match_all('~<td(.*?)\bname="cid\[]"\s+value="(\d+)"(.*?)</tr>~s', $tbody, $matches);
        \preg_match_all('~name="cid\[]"\s+value="(\d+)"~', $tbody, $matches);

        if (!isset($matches[0][0])) {
            return $tbody;
        }

        $db = $this->getDatabase();

        $ids = \array_map([$db, 'quote'], $matches[1]);

        $query = 'SELECT * 
				FROM #__fields_values 
				WHERE field_id IN(' . \implode(',', \array_keys($fields)) . ') 
				    AND item_id IN(' . \implode(',', $ids) . ')';
        $db->setQuery($query);
        $allValues = $db->loadObjectList();

        $itemValues = [];

        foreach ($allValues as $value) {
            // Multiple? Transform to array.
            if (isset($itemValues[$value->item_id][$value->field_id])) {
                if (!\is_array($itemValues[$value->item_id][$value->field_id])) {
                    $itemValues[$value->item_id][$value->field_id] = [$itemValues[$value->item_id][$value->field_id]];
                }

                $itemValues[$value->item_id][$value->field_id][] = $value->value;

                continue;
            }

            // String by default
            $itemValues[$value->item_id][$value->field_id] = $value->value;
        }

        PluginHelper::importPlugin('fields');

        return \preg_replace_callback(
            '~<tr[^>]*>.*?<td.*?\bname="cid\[]"\s+value="(\d+)".*?</tr>~s',
            function ($matches) use ($context, $fields, $itemValues) {
                return $this->decorateListTableBodyRow(
                    $context,
                    $matches[0],
                    $fields,
                    $itemValues[$matches[1]] ?? null
                );
            },
            $html
        );
    }

    /**
     * @param \stdClass[] $fields
     *
     * @since 3.1.0
     */
    protected function decorateListTableBodyRow(
        array $context,
        string $html,
        array $fields,
        ?array $fieldValues
    ): string {
        $columns = [];
        $append = [];

        $dispatcher = $this->application->getDispatcher();

        foreach ($fields as $field) {
            $fieldHtml = '';

            // Set default value
            if (!isset($fieldValues[$field->id])
                && ($field->default_value ?? '') !== ''
                && $field->params->get('af_list_default')
            ) {
                $fieldValues[$field->id] = $field->default_value;
            }

            if (isset($fieldValues[$field->id])) {
                $field->value = $fieldValues[$field->id];
                $field->rawvalue = $field->value;

                // Get rendered output, see FieldsHelper::getFields()

                $item = new \stdClass();

                // Event allow plugins to modify the output of the field before it is prepared
                $dispatcher->dispatch(
                    'onCustomFieldsBeforePrepareField',
                    new BeforePrepareFieldEvent('onCustomFieldsBeforePrepareField', [
                        'context' => $context['context'],
                        'item'    => $item,
                        'subject' => $field,
                    ])
                );

                // Gathering the value for the field
                $value = $dispatcher->dispatch(
                    'onCustomFieldsPrepareField',
                    new PrepareFieldEvent('onCustomFieldsPrepareField', [
                        'context' => $context['context'],
                        'item'    => $item,
                        'subject' => $field,
                    ])
                )->getArgument('result', []);

                if (\is_array($value)) {
                    $value = \array_filter($value, static function ($v) {
                        return $v !== '' && $v !== null;
                    });
                    $value = \implode(' ', $value);
                }

                // Event allow plugins to modify the output of the prepared field
                $value = $dispatcher->dispatch(
                    'onCustomFieldsAfterPrepareField',
                    new AfterPrepareFieldEvent('onCustomFieldsAfterPrepareField', [
                        'context' => $context['context'],
                        'item'    => $item,
                        'subject' => $field,
                        'value'   => &$value,
                        // @todo: Remove reference in Joomla 6, see AfterPrepareFieldEvent::__constructor()
                    ])
                )->getValue();

                // Skip rendering empty values.
                if ($value !== '' && $value !== []) {
                    // Assign the value
                    $field->value = $value;

                    // Some custom preparations
                    $this->prepareFieldBeforeListRender($field);

                    // Use plain output.
                    //$fieldHtml = $value;

                    // Note: frontend template paths are not used in FieldsHelper::render(), only backend template paths.
                    // Hence, use own method with prepended FE template paths.
                    $fieldHtml = $this->renderField(
                        $context['context'],
                        'field.' . $field->params->get('af_layout', 'render'),
                        [
                            'item'    => $item,
                            'context' => $context['context'],
                            'field'   => $field,
                        ]
                    );
                }
            }

            $class = $field->params->get('af_td_class');
            $style = $field->params->get('af_td_style');
            if ($data = $field->params->get('af_td_data_value')) {
                $rawValue = $field->rawvalue ?? '';
                $dataValue = \is_array($rawValue) ? \json_encode($rawValue, \JSON_THROW_ON_ERROR) : $rawValue;
            }

            /** @noinspection PhpUndefinedVariableInspection */
            $fieldHtml = '<td'
                . ($data ? ' data-value="' . \htmlspecialchars($dataValue, \ENT_COMPAT) . '"' : '')
                . ($class ? ' class="' . $class . '"' : '')
                . ($style ? ' style="' . $style . '"' : '') . '>' . $fieldHtml . '</td>';

            if ($field->af_column) {
                $columns[$field->af_column] = $fieldHtml;
            } else {
                $append[] = $fieldHtml;
            }
        }

        if ($columns) {
            $col = 0;

            $html = \preg_replace_callback(
                '~>\s*?((<t[hd])|(</tr))~u',
                static function ($place) use (&$col, &$columns) {
                    while (isset($columns[++$col])) {
                        $new ??= '';
                        $new .= $columns[$col];
                        // just for memory
                        unset($columns[$col]);
                    }

                    return isset($new) ? '>' . $new . $place[1] : $place[0];
                },
                $html
            );
        }

        return $append ? \strtr($html, ['</tr>' => \implode("\n", $append) . '</tr>']) : $html;
    }

    /** @since 1.0 */
    protected function prepareFieldBeforeListRender(\stdClass $field): void
    {
        /** @noinspection DegradedSwitchInspection */
        /** @noinspection PhpSwitchStatementWitSingleBranchInspection */
        switch ($field->type) {
            case 'media':
                $field->value = \strtr($field->value, [
                    ' src="' => ' style="max-width:100%;height:auto" src="' . Uri::root(true) . '/',
                ]);
                break;
        }
    }

    /** @since 1.0 */
    public static function getContext(): array
    {
        if (self::$context === null) {
            $app = Factory::getApplication();

            $map = static::getContextMap();

            $option = $app->getInput()->get('option');

            // Note default empty string required for empty view
            $view = $app->getInput()->get('view', '');

            // Special case for com_categories
            if ($option === 'com_categories') {
                $view = 'categories';
                $extension = $app->getInput()->get('extension', $map['com_categories']['_default'] ?? 'com_content');

                $context = $map[$option][$extension] ?? null;
            } else {
                $context = $map[$option][$view] ?? null;

                // Default view
                if ($view === '' && isset($map[$option][''])) {
                    $view = $map[$option]['_default'];
                }
            }

            if (!isset($context)) {
                return self::$context = [];
            }

            self::$context = [
                'context'    => $context,
                'option'     => $option,
                'view'       => $view,
                'model'      => $map[$option]['_models'][$view] ?? $view,
                'removeCols' => $map[$option]['_removeCols'][$view] ?? null,
            ];
        }

        return self::$context;
    }

    /** @since 1.0 */
    protected static function getContextMap(): array
    {
        $contextMap = [];

        // Prepare map, each line is "fieldsContext|com_option:view1[?],[$view2]"
        foreach (static::$contextMap as $map1) {
            if (empty($map1->enabled)) {
                continue;
            }

            $context = $map1->context;
            $option = $map1->option;
            $view = $map1->view;

            // Custom model name for view
            if (!empty($map1->model)) {
                $contextMap[$option]['_models'][$view] = $map1->model;
            }

            // Default view
            if (\str_ends_with($view, '?')) {
                $view = \substr($view, 0, -1);

                $contextMap[$option][''] = $context;
                $contextMap[$option]['_default'] = $view;
            }

            $contextMap[$option][$view] = $context;

            // Delete columns for view
            if (!empty($map1->removeCols)) {
                $contextMap[$option]['_removeCols'][$view] = \array_flip(
                    \explode(
                        ',',
                        \str_replace(' ', '', \trim($map1->removeCols))
                    )
                );
            }
        }

        return $contextMap;
    }

    /**
     * @return \stdClass[]
     * @since 1.0
     */
    public static function &getFields(?callable $filterCallback = null, ?array $context = null): array
    {
        // Load all fields.
        if (self::$fieldsAll === null) {
            self::$fieldsAll = [];

            // Get all custom field instances
            if ($context ??= static::getContext()) {
                foreach (FieldsHelper::getFields($context['context'], null, false, null, true) as $customField) {
                    static::$fieldsAll[$customField->id] = $customField;
                }
            }
        }

        if (self::$fields === null) {
            self::$fields = [];

            if ($context ??= static::getContext()) {
                /** @var AdministratorApplication $app */
                $app = Factory::getApplication();
                $input = $app->getInput();

                $modelContext = $context['option'] . '.' . $context['view'];
                if ($layout = $input->get('layout')) {
                    $modelContext .= '.' . $layout;
                }
                if ($forcedLanguage = $input->get('forcedLanguage', '')) {
                    $modelContext .= '.' . $forcedLanguage;
                }

                // Detect item category. By default, we provide 0 category (for all categories).
                $item = ['fieldscatid' => 0];
                $filterCategoryId = $app->getUserStateFromRequest(
                    $modelContext . '.filter.category_id',
                    'filter_category_id'
                );

                // Empty filter on filter form reset is not counted.
                if ($filterCategoryId && ($input->getMethod() !== 'POST' || !empty($_POST['filter']['category_id']))) {
                    $item = ['fieldscatid' => $filterCategoryId];
                }

                // All fields, including subform-only fields.
                $fields = [];
                foreach (FieldsHelper::getFields($context['context'], (object)$item, false, [], true) as $field) {
                    $fields[$field->id] = $field;
                }

                // Subform-only fields can be missed if we have selected category, inject from allFields cache.
                foreach ($fields as $field) {
                    // Subform contains children in options.
                    if ($field->type === 'subform') {
                        foreach ($field->fieldparams->get('options', []) as $opt) {
                            if (!isset($fields[$opt->customfield])) {
                                if (!isset(self::$fieldsAll[$opt->customfield])) {
                                    continue;
                                }
                                $fields[$opt->customfield] = self::$fieldsAll[$opt->customfield];
                            }
                        }
                    }
                }

                foreach ($fields as $field) {
                    self::$fields[$field->id] = $field;

                    // Hide display label
                    $field->params->set('showlabel', 0);

                    $field->af_list_title = Text::_(
                        $field->af_list_title ?? $field->params->get('af_list_title', $field->title)
                    );

                    $field->af_listable = $field->params->get('af_listable');
                    $field->af_orderable = $field->params->get('af_orderable');
                    $field->af_searchable = $field->params->get('af_searchable');
                    $field->af_filterable = $field->params->get('af_filterable');
                    $field->af_column = $field->params->get('af_column');

                    // subform and children are not orderable. todo - repeatable is Joomla3 EOL
                    if (!empty($field->only_use_in_subform)
                        || $field->type === 'subform'
                        || $field->type === 'repeatable'
                    ) {
                        $field->af_orderable = false;
                    }
                }
            }
        }

        if (!$filterCallback) {
            return self::$fields;
        }

        $fields = [];
        foreach (self::$fields as $field) {
            if ($filterCallback($field)) {
                $fields[$field->id] = $field;
            }
        }

        return $fields;
    }

    /** @since 1.2.0 */
    protected function decorateFieldList(): void
    {
        $debug = \debug_backtrace(\DEBUG_BACKTRACE_PROVIDE_OBJECT, 5);
        $classname = FieldsModel::class;
        if (!empty($debug[4]['object']) && $debug[4]['object'] instanceof $classname) {
            $this->loadLanguage();
            foreach ($debug[4]['object']->getItems() as $item) {
                $badges = \array_filter([
                    $item->params->get('af_listable') ? Text::_(
                        'PLG_SYSTEM_ALTERFIELDS_BACKEND_FIELDS_INFO_LISTABLE'
                    ) : '',
                    $item->params->get('af_orderable') ? Text::_(
                        'PLG_SYSTEM_ALTERFIELDS_BACKEND_FIELDS_INFO_ORDERABLE'
                    ) : '',
                    $item->params->get('af_filterable') ? Text::sprintf(
                        'PLG_SYSTEM_ALTERFIELDS_BACKEND_FIELDS_INFO_FILTERABLE',
                        $this->getSearchModeTitle($item->params->get('af_filter_mode'))
                    ) : '',
                    $item->params->get('af_searchable') ? Text::sprintf(
                        'PLG_SYSTEM_ALTERFIELDS_BACKEND_FIELDS_INFO_SEARCHABLE',
                        $this->getSearchModeTitle($item->params->get('af_search_mode'))
                    ) : '',
                ]);
                if ($badges) {
                    $item->note = $item->note ? $item->note . '. ' : '';
                    $item->note .= \implode(', ', $badges);
                }
            }
        }
    }

    /** @since 1.2.0 */
    protected function getSearchModeTitle(?string $mode): string
    {
        return $mode ? Text::_('PLG_SYSTEM_ALTERFIELDS_FIND_MODE_' . $mode) : 'n/a';
    }

    /**
     * Duplicates original method but prepends frontend template paths missed in original code.
     * @see   FieldsHelper::render
     * @since 2.1.1+
     */
    protected function renderField(string $context, string $layoutFile, array $displayData): ?string
    {
        $value = '';

        $prependsPaths = [];

        /*
         * Because the layout refreshes the paths before the render function is
         * called, so there is no way to load the layout overrides in the order
         * template -> context -> fields.
         * If there is no override in the context then we need to call the
         * layout from Fields.
         */
        if ($parts = self::extractFieldContext($context)) {
            // Trying to render the layout on the component from the context
            $value = $this->renderFieldLayout(
                $layoutFile,
                $displayData,
                null,
                $prependsPaths + ['component' => $parts[0], 'client' => 0]
            );
        }

        if (($value ?? '') === '') {
            // Trying to render the layout on Fields itself
            $value = $this->renderFieldLayout(
                $layoutFile,
                $displayData,
                null,
                $prependsPaths + ['component' => 'com_fields', 'client' => 0]
            );
        }

        return $value;
    }

    /** @since 2.1.1+ */
    protected function renderFieldLayout(
        string $layoutFile,
        ?array $displayData,
        ?string $basePath,
        array $options
    ): ?string {
        static $includePaths = null;

        if ($includePaths === null) {
            $db = $this->getDatabase();
            $query = $db->createQuery();

            // Build the query.
            $query->select('element, name')
                ->from('#__extensions')
                ->where($db->quoteName('client_id') . ' = 0')
                ->where($db->quoteName('type') . ' = ' . $db->quote('template'))
                ->where($db->quoteName('enabled') . ' = 1');

            // Set the query and load the templates.
            $db->setQuery($query);
            $templates = $db->loadObjectList('element');

            $extension = $this->application->getInput()->get('option');

            $includePaths = [];
            foreach ($templates as $template) {
                $includePaths = \array_merge($includePaths, [
                    Path::clean(JPATH_SITE . '/templates/' . $template->element . '/html/layouts/' . $extension),
                    Path::clean(JPATH_SITE . '/templates/' . $template->element . '/html/layouts/com_fields'),
                    Path::clean(JPATH_SITE . '/templates/' . $template->element . '/html/layouts'),
                ]);
            }
        }

        $layout = new FileLayout($layoutFile, $basePath, $options);

        $layout->addIncludePaths($includePaths);

        return $layout->render($displayData);
    }

    /**
     * Copy from Joomla code.
     * @see     FieldsHelper::extract
     * @since   2.1.1
     */
    public static function extractFieldContext(?string $contextString, ?object $item = null): ?array
    {
        if ($contextString === null) {
            return null;
        }

        $parts = \explode('.', $contextString, 2);

        if (\count($parts) < 2) {
            return null;
        }

        $newSection = '';

        $component = Factory::getApplication()->bootComponent($parts[0]);

        if ($component instanceof FieldsServiceInterface) {
            $newSection = $component->validateSection($parts[1], $item);
        }

        if ($newSection) {
            $parts[1] = $newSection;
        }

        return $parts;
    }
}
