<?php
/**
 * @package        Alter Fields
 * @copyright      Copyright (C) 2022-2023 AlterBrains.com. All rights reserved.
 * @license        https://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL
 */

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\Database\DatabaseInterface;

defined('_JEXEC') or die;

/**
 * @since        1.0
 * @noinspection PhpUnused
 */
class plgSystemAlterFieldsInstallerScript
{
    /**
     * @var string
     * @since 3.0
     */
    protected $extension_name = 'System - Alter Fields';

    /**
     * @var string
     * @since 2.0
     */
    protected $minimumPhp = '8.3';

    /**
     * @var string
     * @since 2.0
     */
    protected $minimumJoomla = '5.4';

    /**
     * @return bool
     * @since 2.0
     */
    public function preflight()
    {
        if (!empty($this->minimumPhp) && version_compare(PHP_VERSION, $this->minimumPhp, '<')) {
            Factory::getApplication()->enqueueMessage(Text::sprintf('JLIB_INSTALLER_MINIMUM_PHP', $this->minimumPhp), 'error');

            return false;
        }
        if (!empty($this->minimumJoomla) && version_compare(JVERSION, $this->minimumJoomla, '<')) {
            Factory::getApplication()->enqueueMessage(Text::sprintf('JLIB_INSTALLER_MINIMUM_JOOMLA', $this->minimumJoomla), 'error');

            return false;
        }

        return true;
    }

    public function install()
    {
        Factory::getApplication()->enqueueMessage(sprintf('Successfully installed "%s" plugin.', $this->extension_name));

        $defaultMap = [
            'com_users.user|com_users:users?',
            'ccom_content.article|com_content:articles?',
            'com_content.categories|com_categories:com_content?',
            'com_contact.contact|com_contact:contacts?',
            'com_contact.categories|com_categories:com_contact',
            'com_areports.report|com_areports:reports?',
            'com_areports.categories|com_categories:com_areports',
            'com_asurveys.survey|com_asurveys:surveys?',
            'com_asurveys.categories|com_categories:com_asurveys',
            'com_aglossary.article|com_aglossary:articles?',
            'com_aglossary.categories|com_aglossary:com_aglossary',
            'com_dpcalendar.event|com_dpcalendar:events>AdminEvents',
            'com_dpcalendar.location|com_dpcalendar:locations>Locations',
        ];

        // Set default map
        $this->updateContextMap(implode("\n", $defaultMap));
    }

    public function update()
    {
        Factory::getApplication()->enqueueMessage(sprintf('Successfully updated "%s" plugin.', $this->extension_name));

        // Update old map before 3.1.0
        $this->updateContextMap();
    }

    public function uninstall()
    {
        Factory::getApplication()->enqueueMessage(sprintf('Successfully uninstalled "%s" plugin.', $this->extension_name));
    }

    /**
     * @since 3.1.0+
     */
    protected function updateContextMap(?string $oldMap = null): void
    {
        /** @var DatabaseInterface::class $db */
        $db = Factory::getContainer()->get(DatabaseInterface::class);

        $pluginName = strtolower(strtr(__CLASS__, [
            'plgSystem' => '',
            'InstallerScript' => '',
        ]));

        // Load directly from DB, but not via PluginHelper because plugin can be disabled.
        $query = $db->getQuery(true)
            ->select('extension_id, params')
            ->from('#__extensions')
            ->where('type = "plugin"')
            ->where('folder = "system"')
            ->where('element = "' . $pluginName . '"');
        $db->setQuery($query);
        $plugin = $db->loadObject();

        if (!$plugin) {
            return;
        }

        $params = new Joomla\Registry\Registry($plugin->params);

        $oldMap ??= $params->get('contextMap');
        /** @noinspection CallableParameterUseCaseInTypeContextInspection */
        if (!$oldMap || is_array($oldMap) || is_object($oldMap)) {
            return;
        }

        $new = [];

        $contextMapData = explode(
            "\n",
            strtr(trim($oldMap), [
                "\r\n" => "\n",
                ' '    => '',
            ])
        );

        foreach ($contextMapData as $map1) {
            if (('' === $map1 = trim($map1)) || !str_contains($map1, '|')) {
                continue;
            }

            [$context, $componentView] = @explode('|', $map1, 2);

            if (str_contains($componentView, ':')) {
                [$component, $views] = explode(':', $componentView, 2);

                foreach (explode(',', $views) as $view) {
                    $model = '';

                    // Custom model name for view
                    if (str_contains($view, '>')) {
                        [$view, $model] = explode('>', $view, 2);
                    }

                    $new['contextMap' . count($new)] = [
                        'enabled' => '1',
                        'context' => $context,
                        'option'  => $component,
                        'view'    => $view,
                        'model'   => $model,
                    ];
                }
            }
        }

        $params->set('contextMap', $new);

        $query = $db->getQuery(true)
            ->update('#__extensions')
            ->set('params = ' . $db->quote($params->toString()))
            ->where('extension_id = ' . $plugin->extension_id);
        $db->setQuery($query);
        $db->execute();
    }
}
