/**
 * @package      AJAX Toggler
 * @copyright    Copyright (C) 2009 - 2023 AlterBrains.com. All rights reserved.
 * @license      Commercial
 */
'use strict';

// noinspection JSUnusedLocalSymbols,JSUnresolvedVariable
window.AjaxToggler = {
	options: {
		base: null,
		menu: true,
		toolbarTasks: ['publish', 'unpublish', 'featured', 'unfeatured', 'archive', 'checkin', 'trash', 'delete', 'rebuild', 'duplicate', 'refresh', 'block', 'unblock', 'activate'],
		edit: true
	},

	/**
	 * @var {HTMLFormElement|node}
	 */
	form: null,
	table: null,
	wrapper: null,
	overlay: null,

	current_task: null,
	flag: null,
	flag_ordering: null,
	jsTools: null,

	initialize(options) {
		Object.assign(this.options, options);

		// noinspection JSUnresolvedVariable
		if (window.Joomla && document.adminForm && this.parseTableForm(document.adminForm)) {
			this.parseJoomla();
		}
	},

	/**
	 * @param {HTMLFormElement} form
	 * @return {boolean}
	 */
	parseTableForm(form) {
		if (!form || !form.task || !form.boxchecked) {
			return false;
		}

		this.form = form;

		this.table = document.querySelector('#j-main-container > table.table, #j-main-container > div > table.table');

		if (!this.table) {
			return false;
		}

		this.current_task = form.task.value;

		// create flag
		if (typeof (this.form['jatoggler']) === 'undefined') {
			this.flag = document.createElement('input');
			this.flag.type = 'hidden';
			this.flag.name = 'jatoggler';
			this.flag.value = '0';
			this.form.appendChild(this.flag);
		}
		if (typeof (this.form['jatoggler_ordering']) === 'undefined') {
			this.flag_ordering = document.createElement('input');
			this.flag_ordering.type = 'hidden';
			this.flag_ordering.name = 'jatoggler_ordering';
			this.flag_ordering.value = '';
			this.form.appendChild(this.flag_ordering);
		}

		return true;
	},

	parseJoomla() {
		if (!window.Joomla._ajaxtoggler) {
			window.Joomla._ajaxtoggler = true;

			// Custom submitform
			window.Joomla._submitform = window.Joomla.submitform;
			window.Joomla.submitform = (task, form, validate) => {
				this.submitTableForm(task, form, validate);
			};

			// Custom tableOrdering
			window.Joomla._tableOrdering = window.Joomla.tableOrdering;
			window.Joomla.tableOrdering = (order, dir, task, form) => {
				this.tableOrdering(order, dir, task, form);
			};

			// Custom listItemTask
			window.Joomla._listItemTask = window.Joomla.listItemTask;
			window.Joomla.listItemTask = (id, task, form = null) => {
				return this.listItemTask(id, task, form);
			};
		}

		// Joomla 3.2.1+, 4.x Table ordering
		this.jsTools = document.querySelector('.js-stools');

		if (this.jsTools) {
			// Capture click on TH to prevent any Joomla events on a.click
			document.querySelectorAll('a.js-stools-column-order').forEach((a) => {
				a.parentNode.addEventListener('click', (e) => {
					e.preventDefault();
					e.stopImmediatePropagation();

					let target = e.target;

					// can be executed for inner element, so check for A
					if (e.target.tagName !== 'A') {
						target = e.target.closest('th').querySelector('a');
					}
					console.log(target)

					if (target) {
						const order = target.getAttribute('data-order');
						const dir = target.getAttribute('data-direction');

						const listFullOrdering = document.querySelector('#list_fullordering');
						if (listFullOrdering) {
							listFullOrdering.value = order + ' ' + dir;
							const event = new Event('liszt:updated');
							listFullOrdering.dispatchEvent(event);
						}

						this.tableOrdering(order, dir, '', null);
					}
				}, true);
			});
		}
	},

	createWrapper() {
		if (!this.wrapper) {
			this.wrapper = document.createElement('div');
			this.wrapper.id = 'ajaxtogglerWrapper';
			this.wrapper.style.clear = 'both';
			this.table.parentNode.insertBefore(this.wrapper, this.table);
			this.wrapper.appendChild(this.table);
		}
	},

	/**
	 * @param {HTMLElement} parent
	 */
	createOverlay(parent) {
		document.body.style.cursor = 'wait';

		// Custom case, we need to adopt native Joomla layer a bit
		if (parent) {
			parent.style.position = 'relative';

			const layer = document.createElement('joomla-core-loader');
			parent.prepend(layer);
			layer.style.position = 'absolute';
			layer.style.bottom = '0';
			layer.style.backgroundColor = 'inherit';

			// prettify wrapper
			if (parent === this.wrapper) {
				this.wrapper.style.overflow = 'hidden';
				this.wrapper.style.zIndex = '1';
				this.wrapper.style.position = 'relative';
			}
		}
	},

	removeOverlay() {
		document.body.style.cursor = '';

		// Remove wrapper style if any.
		if (this.wrapper) {
			this.wrapper.style.overflow = '';
			this.wrapper.style.zIndex = '';
			this.wrapper.style.position = '';
		}
	},

	/**
	 * Generic submit form
	 *
	 * @param  {String}  task      The given task
	 * @param  {node}    form      The form element
	 * @param  {bool}    validate  The form element
	 *
	 * @returns  {boolean} Note: we need to return Bool here, not like in Joomla!
	 */
	submitTableForm(task, form, validate) {
		// usual submit if we have task
		if (typeof (task) !== 'undefined' && task !== '') {
			let realTask = task;
			if (realTask.indexOf('.') !== -1) {
				realTask = realTask.split('.')[1];
				if (this.options.toolbarTasks.includes(realTask)) {
					this.form.task.value = task;
					// noinspection JSIgnoredPromiseFromCall
					this.loadTable();
					return false;
				}
			}

			this.form.task.value = task;

			// Submit the form.
			if (typeof this.form.onsubmit === 'function') {
				this.form.onsubmit();
			}
			if (typeof this.form.fireEvent === 'function') {
				this.form.fireEvent('submit');
			}
			this.form.submit();
			return false;
		}

		// noinspection JSIgnoredPromiseFromCall
		this.loadTable();

		return false;
	},

	/**
	 * USED IN: libraries/joomla/html/html/grid.php
	 * In other words, on any reorderable table
	 *
	 * @param  {string}  order  The order value
	 * @param  {string}  dir    The direction
	 * @param  {string}  task   The task
	 * @param  {node}    form   The form
	 *
	 * @returns  {boolean} Note: we need to return Bool here, not like in Joomla!
	 */
	tableOrdering(order, dir, task, form) {
		if (typeof (form) === 'undefined' || form === null) {
			form = this.form;
		}

		// noinspection JSUnresolvedVariable
		if (form.filter_order !== undefined) {
			// noinspection JSUnresolvedVariable
			this.form.filter_order.value = order;
		}
		// noinspection JSUnresolvedVariable
		if (form.filter_order_Dir !== undefined) {
			// noinspection JSUnresolvedVariable
			this.form.filter_order_Dir.value = dir;
		}

		const sortTable = document.querySelector('#sortTable');
		const directionTable = document.querySelector('#directionTable');

		if (sortTable) {
			sortTable.value = order;
			const event = new Event('liszt:updated');
			sortTable.dispatchEvent(event);
		}

		if (directionTable) {
			directionTable.value = dir;
			const event = new Event('liszt:updated');
			directionTable.dispatchEvent(event);
		}

		this.form.task.value = task;

		// noinspection JSIgnoredPromiseFromCall
		this.loadTable();

		return false;
	},

	/**
	 * USED IN: all over :)
	 *
	 * @param  {string}  id    The id
	 * @param  {string}  task  The task
	 * @param  {HTMLFormElement}  form  The form
	 *
	 * @return {boolean}
	 */
	listItemTask(id, task, form = null) {
		if (form === null) {
			form = this.form;
		}
		const cb = form[id];
		if (cb) {
			for (let i = 0; true; i++) {
				let cbx = form['cb' + i];
				if (!cbx)
					break;
				cbx.checked = false;
			}
			cb.checked = true;
			form.boxchecked.value = 1;
			form.task.value = task;
			// noinspection JSIgnoredPromiseFromCall
			this.loadTable();
		}

		return false;
	},

	loadTable() {
		this.createWrapper();
		this.createOverlay(this.wrapper);
		this.flag.value = '1';

		// Is ordering loaded?
		this.flag_ordering.value = window.dragula ? '1' : '';

		// Disable activated list-required buttons
		document.querySelectorAll('joomla-toolbar-button[list-selection]').forEach(el => {
			el.setDisabled(true);
		});

		// Collect all form data including hidden fields
		const formData = new FormData();

		// Add all form elements
		const elements = this.form.elements;
		for (let i = 0; i < elements.length; i++) {
			const element = elements[i];
			if (element.name) {
				if (element.type === 'checkbox' || element.type === 'radio') {
					if (element.checked) {
						formData.append(element.name, element.value);
					}
				} else if (element.type === 'select-multiple') {
					const selectedOptions = Array.from(element.selectedOptions);
					selectedOptions.forEach(option => {
						formData.append(element.name, option.value);
					});
				} else {
					formData.append(element.name, element.value);
				}
			}
		}

		// Convert FormData to URLSearchParams
		const searchParams = new URLSearchParams();
		for (const [key, value] of formData) {
			searchParams.append(key, value);
		}

		return fetch(this.form.action, {
			method: 'POST',
			headers: {
				'Content-Type': 'application/x-www-form-urlencoded',
				'X-Requested-With': 'XMLHttpRequest'
			},
			body: searchParams.toString()
		})
			.then(response => response.json())
			.then((resp) => {
				const tempDiv = document.createElement('div');
				tempDiv.innerHTML = resp.data.html;

				const scripts = tempDiv.querySelectorAll('script');
				scripts.forEach(script => script.remove());

				const stylesheets = tempDiv.querySelectorAll('link[rel="stylesheet"], style');
				stylesheets.forEach(stylesheet => stylesheet.remove());

				// Main table
				document.querySelector('#ajaxtogglerWrapper').innerHTML = tempDiv.innerHTML;

				// Load stylesheets
				stylesheets.forEach(style => {
					if (style.tagName === 'LINK') {
						const newLink = document.createElement('link');
						newLink.rel = 'stylesheet';
						newLink.href = style.href;
						document.head.appendChild(newLink);
					} else {
						document.head.appendChild(style.cloneNode(true));
					}
				});

				// Execute scripts, one by one
				function loadScriptsSequentially(scripts, index = 0) {
					if (index >= scripts.length) {
						return;
					}

					const oldScript = scripts[index];
					const newScript = document.createElement('script');

					if (oldScript.src) {
						newScript.src = oldScript.src;
						newScript.onload = () => loadScriptsSequentially(scripts, index + 1);
						newScript.onerror = () => loadScriptsSequentially(scripts, index + 1);
					} else {
						newScript.textContent = oldScript.textContent;
						loadScriptsSequentially(scripts, index + 1);
					}

					document.head.appendChild(newScript);
				}
				loadScriptsSequentially([...scripts]);

				// Toolbar
				if (resp.data.toolbar) {
					const toolbar = document.querySelector('#toolbar');
					if (toolbar && toolbar.parentNode) {
						toolbar.parentNode.outerHTML = resp.data.toolbar;
					}
				}

				// Pagination
				const pagination = document.querySelector('nav.pagination__wrapper');
				if (pagination) {
					// noinspection JSUnresolvedReference
					pagination.innerHTML = resp.data.pagination;
				}

				// Toolbar title
				if (resp.data.title) {
					const title = document.querySelector('#header .container-title');
					if (title) {
						title.innerHTML = resp.data.title;
					}
				}

				// Messages
				document.getElementById('system-message-container').innerHTML = '';

				if (resp.messages) {
					// noinspection JSCheckFunctionSignatures
					Joomla.renderMessages(resp.messages);
				}

				this.flag.value = '0';
				this.form.task.value = this.current_task;
				this.form.boxchecked.value = '';

				this.removeOverlay();

				this.parseJoomla();
				this.checkScroll();

				this.checkContext();
				this.checkTooltips();
				this.checkPopovers();
				this.checkMultiselect();
				this.checkScripts();
			})
			.catch(() => {
				this.flag.value = '0';
				this.form.task.value = this.current_task;
				this.removeOverlay();
			});
	},

	checkScroll() {
		const formRect = this.form.getBoundingClientRect();
		if (formRect.bottom < window.scrollY) {
			window.scroll(0, 0);
		}
	},

	checkTooltips() {
		// Remove MooTools tip
		document.querySelectorAll('.tip-wrap').forEach(el => el.remove());

		// Remove Bootstrap tip
		document.querySelectorAll('div.tooltip.fade.top').forEach(el => el.remove());

		// Bootstrap tooltips
		if (typeof bootstrap !== 'undefined' && bootstrap.Tooltip) {
			document.querySelectorAll('#' + this.wrapper.id + ' .hasTooltip, [rel=tooltip]').forEach(el => {
				new bootstrap.Tooltip(el, {
					html: true,
					container: 'body'
				});
			});
		}
	},

	checkPopovers() {
		if (typeof bootstrap !== 'undefined' && bootstrap.Popover) {
			// Remove Bootstrap popovers
			document.querySelectorAll('div.popover.fade.top').forEach(el => el.remove());

			const options = {
				animation: true,
				container: 'body',
				delay: {
					show: 50,
					hide: 200
				},
				html: true,
				trigger: 'hover focus',
				offset: [0, 10],
				boundary: 'scrollParent'
			};

			// Bootstrap popovers
			document.querySelectorAll('#' + this.wrapper.id + ' .hasPopover, [rel=popover]').forEach(el => {
				new bootstrap.Popover(el, options);
			});
		}
	},

	checkContext() {
		document.querySelectorAll('#' + this.wrapper.id + ' .has-context').forEach(el => {
			el.addEventListener('mouseenter', function () {
				const btnGroup = this.querySelector('.btn-group');
				if (btnGroup) btnGroup.style.display = 'block';
			});

			el.addEventListener('mouseleave', function () {
				const btnGroup = this.querySelector('.btn-group');
				if (btnGroup) {
					btnGroup.style.display = 'none';
					btnGroup.classList.remove('open');
				}
			});
		});
	},

	checkMultiselect() {
		// Just disable actions by default.
		document.querySelectorAll('joomla-toolbar-button[list-selection]').forEach(el => {
			el.setDisabled(true);
			el.querySelectorAll('.dropdown-menu').forEach(el2 => {
				el2.classList.remove('show');
			});
		});
	},

	checkScripts() {
		if (document.querySelectorAll('.js-grid-item-action').length) {
			for (let s of document.scripts) {
				if (s.src.indexOf('/media/system/js/list-view.') !== -1) {
					this.rerunModule(s);
				} else if (s.src.indexOf('/media/system/js/table-columns.') !== -1) {
					// Apply column order only into our table.
					[...document.querySelectorAll('table:not(.columns-order-ignore)')].forEach(t => {
						if (t.parentNode.id !== 'ajaxtogglerWrapper') {
							t.classList.add('columns-order-ignore');
						}
					});
					// Remove existing button menu
					const dropdown = this.wrapper.previousElementSibling;
					if (dropdown && dropdown.matches('.dropdown.float-end.pb-2')) {
						dropdown.remove();
					}
					this.rerunModule(s);
				} else if (s.src.indexOf('/media/system/js/multiselect.') !== -1) {
					const old = document.addEventListener;
					document.addEventListener = (e, f) => {
						if (e === 'DOMContentLoaded') {
							f();
							document.addEventListener = old;
						}
					}
					this.rerunModule(s);
				}
			}
		}
	},

	/**
	 * @param {HTMLScriptElement} s
	 */
	rerunModule(s) {
		fetch(s.src).then(async r => {
			const ns = document.createElement('script');
			ns.type = 'module';
			ns.textContent = await r.text();
			document.body.prepend(ns);
		});
	}
};
