<?php
/**
 * @package        AJAX Toggler
 * @copyright      Copyright (C)  2009 - 2023 AlterBrains.com. All rights reserved.
 * @license        GNU/GPL, see LICENSE.txt
 */

/** @noinspection PhpMultipleClassDeclarationsInspection */

namespace AlterBrains\Plugin\System\Ajaxtoggler\Extension;

use Joomla\CMS\Application\AdministratorApplication;
use Joomla\CMS\Application\SiteApplication;
use Joomla\CMS\Document\HtmlDocument;
use Joomla\CMS\Environment\Browser;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Response\JsonResponse;
use Joomla\CMS\Session\Session;
use Joomla\CMS\Toolbar\Toolbar;
use Joomla\CMS\Uri\Uri;
use Joomla\Event\SubscriberInterface;

\defined('_JEXEC') or die;

/**
 * @since 1.0
 */
class Ajaxtoggler extends CMSPlugin implements SubscriberInterface
{
    /**
     * @var SiteApplication|AdministratorApplication
     * @since        1.0
     * @noinspection PhpMissingFieldTypeInspection
     */
    private $application;

    /**
     * @var bool
     * @since 3.2.0
     */
    protected bool $active = false;

    /**
     * @since 3.2.3
     */
    protected HtmlDocument $document;

    /**
     * @since 3.2.3
     */
    protected ?Session $session = null;

    /**
     * @inheritDoc
     * @since 4.0
     */
    public function __construct($dispatcher, array $config = [])
    {
        parent::__construct($dispatcher, $config);

        $this->application = Factory::getApplication();
    }

    /**
     * @inheritDoc
     * @since 1.0
     */
    public static function getSubscribedEvents(): array
    {
        return [
            'onAfterRoute'        => 'onAfterRoute',
            'onAfterDispatch'     => 'onAfterDispatch',
            'onBeforeCompileHead' => 'onBeforeCompileHead',
        ];
    }

    public function onAfterRoute(): void
    {
        if (\in_array($this->application->getInput()->get('option'), $this->params->get('exclude', []), true)) {
            return;
        }

        $this->session = $this->application->getSession();

        if ($this->application->getInput()->getBool('jatoggler')) {
            $this->session->set('jatoggler', 1);

            // We should normally redirect in $app->redirect, else we fail with stupid IE
            $navigator = Browser::getInstance();
            if ($navigator->isBrowser('msie')) {
                $navigator->setBrowser('chrome');
            }
        }
    }

    /**
     * @since 1.0
     */
    public function onAfterDispatch(): void
    {
        if (!$this->session || $this->application->getDocument()->getType() !== 'html') {
            return;
        }

        /** @noinspection PhpUnhandledExceptionInspection */
        $this->document = $this->application->getDocument();

        $this->active = true;

        if ($this->session->get('jatoggler')) {
            $this->session->set('jatoggler', 0);

            if (\strtolower(
                    $this->application->getInput()->server->get('HTTP_X_REQUESTED_WITH', '')
                ) === 'xmlhttprequest') {
                echo new JsonResponse([
                    'html'       => $this->getTable() . $this->getOrdering(),
                    'toolbar'    => $this->getToolbar(),
                    'pagination' => $this->getPagination(),
                    'title'      => $this->application->JComponentTitle ?? null,
                ]);

                $this->application->close();
            }
        }
    }

    /**
     * @since 3.3.0
     */
    public function onBeforeCompileHead(): void
    {
        if (!$this->active) {
            return;
        }

        $options = \json_encode([
            'base' => Uri::root(),
        ], \JSON_THROW_ON_ERROR);

        /** @noinspection BadExpressionStatementJS */
        /** @noinspection JSVoidFunctionReturnValueUsed */
        $initScript = <<<JS
document.addEventListener('DOMContentLoaded', ()=>{AjaxToggler.initialize($options)});
JS;

        $script = $this->_name . (\JDEBUG ? '.js' : '.min.js');

        $this->document->getWebAssetManager()
            ->registerAndUseScript(
                'plg_' . $this->_name . '.admin',
                'plugins/system/' . $this->_name . '/media/js/' . $script,
                ['version' => 'auto'],
                ['defer' => false],
                ['core']
            )
            ->addInlineScript($initScript)
            ->useScript('webcomponent.core-loader');
    }

    /**
     * @since 1.0
     */
    protected function getTable(): string
    {
        $html = '';

        // We have table
        // Apply lazy match to capture the full table, use negative lookahead
        \preg_match_all('/(<table((?!<table>)).*?<\/table>)/s', $this->document->getBuffer('component'), $matches);

        if (!empty($matches[1])) {
            foreach ($matches[1] as $table) {
                if (
                    // Should contain all toggle
                    \str_contains($table, '<input class="form-check-input" autocomplete="off"')
                    // We can have multiple tables, use the biggest one
                    && \strlen($html) < \strlen($table)
                    && (
                        // Usual Joomla table
                        \str_contains($table, ' class="table')
                        // K2
                        || \str_contains($table, ' class="adminlist table table-striped')
                    )
                ) {
                    $html = $table;
                }
            }
        }

        // No results! Keep our wrapper ID!
        if ($html === '') {
            $html = '<div id="ajaxtogglerWrapper" class="alert alert-info">' .
                '<span class="icon-info-circle" aria-hidden="true"></span><span class="visually-hidden">' . Text::_(
                    'INFO'
                ) . '</span> ' .
                Text::_('JGLOBAL_NO_MATCHING_RESULTS') .
                '</div>';
        }

        // small house-keeping
        return $html;
    }

    /**
     * @since 1.0
     */
    protected function getToolbar(): string
    {
        $html = '';

        // Reload toolbar if Trashed state changed
        $state1 = $this->session->get('jatoggler_filter_published');
        $state2 = $this->application->getInput()->get(
            'filter_published',
            $this->application->getInput()->get('filter_state')
        );

        // New Joomla 3.2.1+ filter
        $filter = $this->application->getInput()->get('filter', [], 'array');

        if (isset($filter['published'])) {
            $state2 = $filter['published'];
        } elseif (isset($filter['state'])) {
            $state2 = $filter['state'];
        }

        /** @noinspection TypeUnsafeComparisonInspection */
        if (
            // Reload toolbar if Trashed state changed
            (($state1 == -2 || $state2 == -2) && $state1 != $state2)
            ||
            // Reload toolbar for selected components;
            ($this->params->get('toolbar')
                && \in_array($this->application->getInput()->get('option'), $this->params->get('toolbar', []), true)
            )
        ) {
            $this->session->set('jatoggler_filter_published', $state2);

            // render toolbar
            /** @noinspection PhpDeprecationInspection */
            /** @noinspection PhpRedundantOptionalArgumentInspection */
            $html = Toolbar::getInstance('toolbar')->render();
        }

        return $html;
    }

    /**
     * @since 1.0
     */
    protected function getPagination(): string
    {
        $html = '';

        $pattern = '/<nav class="pagination__wrapper[^>]+>(.*?)<\/nav>/Us';

        if (\preg_match_all($pattern, $this->document->getBuffer('component'), $matches) && !empty($matches[1][0])) {
            $html = $matches[1][0];
        }

        return $html;
    }

    /**
     * Re-init ordering list if table was ordered by ordering.
     * @since 1.0
     */
    protected function getOrdering(): string
    {
        // Already loaded
        $exists = $this->application->getInput()->getBool('jatoggler_ordering');

        $html = '';

        /** @noinspection PhpUnhandledExceptionInspection */
        $ordering = $this->application->getInput()->get('filter_order');

        // New Joomla 3.2.1+ list
        $list = $this->application->getInput()->get('list', [], 'array');

        if (isset($list['fullordering'])) {
            $ordering = \strtr($list['fullordering'], [
                ' ASC'  => '',
                ' DESC' => '',
            ]);
        }

        if (\in_array($ordering, ['a.ordering', 'ordering', 'a.lft', 'lft'], true)) {
            if (!$exists) {
                $html .= $this->extractScript('dragula');
                $html .= $this->extractStylesheet('dragula');
            }

            // Maximum perversion.
            //$html .= $this->extractScript('joomla.draggable');
            if (
                ($item = $this->document->getWebAssetManager()->getAsset('script', 'joomla.draggable'))
                && ($file = JPATH_SITE . '/' . $item->getUri())
                && \is_file($file)
            ) {
//                $time = \time();
//
//                // Wrap into function to prevent "Uncaught SyntaxError: Identifier 'url' has already been declared"
//                $html .= '<script>'
//                    . \strtr(
//                        '(function () {' . \file_get_contents($file) . '}());',
//                        ['DOMContentLoaded' => 'DOMContentLoaded' . $time]
//                    )
//                    . '; document.dispatchEvent(new CustomEvent("DOMContentLoaded' . $time . '", {}));'
//                    . '</script>';

                // Wrap into function to prevent "Uncaught SyntaxError: Identifier 'url' has already been declared"
                $html .= '<script>'
                    . 'function AjaxTogglerOrdering() {' . \file_get_contents($file) . '}'
                    . 'AjaxTogglerOrdering();'
                    . '</script>';
            }

            // todo, allow custom options. Joomla doesn't add them by default.
            //$opts = $this->document->getScriptOptions('draggable-list');
        }

        return $html;
    }

    /**
     * @since 3.3.0
     */
    protected function extractScript(string $name): string
    {
        if ($item = $this->document->getWebAssetManager()->getAsset('script', $name)) {
            $uri = $item->getUri();
            if ($version = $item->getVersion()) {
                $uri .= (!\str_contains('?', $uri) ? '?' : '&amp;') . $version;
            }

            return '<script src="' . $uri . '"></script>';
        }

        return '';
    }

    /**
     * @since 3.3.0
     */
    protected function extractStylesheet(string $name): string
    {
        if ($item = $this->document->getWebAssetManager()->getAsset('style', $name)) {
            $uri = $item->getUri();
            if ($version = $item->getVersion()) {
                $uri .= (!\str_contains('?', $uri) ? '?' : '&amp;') . $version;
            }

            return '<link href="' . $uri . '" rel="stylesheet" />';
        }

        return '';
    }
}
