<?php
/**
 * @package        Direct Alias
 * @copyright      Copyright (C) 2009-2023 AlterBrains.com. All rights reserved.
 * @license        https://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL
 */

/** @noinspection PhpMultipleClassDeclarationsInspection */

namespace AlterBrains\Plugin\System\Directalias\Extension;

use Joomla\CMS\Application\SiteApplication;
use Joomla\CMS\Event\Model;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Form\FormHelper;
use Joomla\CMS\Menu\SiteMenu;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Router\Router;
use Joomla\CMS\Router\SiteRouter;
use Joomla\CMS\Uri\Uri;
use Joomla\Event\Event;
use Joomla\Event\SubscriberInterface;

\defined('_JEXEC') or die;

/**
 * @since        3.0.0
 * @noinspection PhpUnused
 */
class Directalias extends CMSPlugin implements SubscriberInterface
{
    /**
     * @var bool
     * @since 2.1.1
     */
    private static $has_falang;
    /**
     * @var   ?SiteRouter
     * @since 3.0.0
     */
    private $router;
    /**
     * @var array
     * @since 3.0.0
     */
    private $custom_routes = [];

    /**
     * @inheritDoc
     * @since 3.0.0
     */
    public static function getSubscribedEvents(): array
    {
        static::$has_falang = \class_exists(\plgSystemFalangdriver::class);

        // onAfterInitialise is executed in Administrator as well because site routes can be generated in backend.
        $events = [
            // Use the lowest priority to execute Falang first, highest otherwise.
            'onAfterInitialise' => ['onAfterInitialise', static::$has_falang ? -1 : \PHP_INT_MAX],
        ];

        if (Factory::getApplication()->isClient('administrator')) {
            return $events + [
                    'onContentPrepareForm' => 'onContentPrepareForm',
                ];
        }

        return $events;
    }

    /**
     * @since  3.0.0
     */
    public function onContentPrepareForm(Event/*Model\PrepareFormEvent */ $event): void
    {
        /**
         * @var Form  $form
         * @var mixed $data
         */
        [$form/*, $data*/] = $event instanceof Model\PrepareFormEvent ? [
            $event->getForm()
            /*, $event->getData()*/
        ] : $event->getArguments();

        if ($form->getName() !== 'com_menus.item' || $this->params->get('shorten_all')) {
            return;
        }

        $this->loadLanguage();

        FormHelper::addFieldPrefix('AlterBrains\Plugin\System\\' . (new \ReflectionClass($this))->getShortName() . '\Field');

        $form->setFieldAttribute('alias', 'type', 'directalias');

        $form->load(
            '<?xml version="1.0" encoding="utf-8"?>
			<form>
				<fields name="params">
					<fieldset name="menu-options">
						<field name="direct_alias" type="hidden" />
						<field name="absent_alias" type="hidden" />
					</fieldset>
				</fields>
			</form>',
            false
        );

        // Display real switchers in Falang
        if ($this->getApplication()->input->get('option') === 'com_falang') {
            $form->load(
                '<?xml version="1.0" encoding="utf-8"?>
				<form>
					<fields name="params">
						<fieldset name="menu-options">
							<!--suppress HtmlUnknownAttribute -->
							<field name="direct_alias" type="radio" class="btn-group btn-group-yesno" default="0"
							    label="PLG_SYSTEM_FIELD_DIRECT_ALIAS_MODE" description="PLG_SYSTEM_DIRECT_ALIAS_DIRECT_TIP_DESC">
								<option value="1">PLG_SYSTEM_DIRECT_ALIAS_DIRECT</option>
								<option value="0">PLG_SYSTEM_DIRECT_ALIAS_RELATIVE</option>
							</field>
							<!--suppress HtmlUnknownAttribute -->
							<field name="absent_alias" type="radio" class="btn-group btn-group-yesno" default="0"
							    label="PLG_SYSTEM_FIELD_ABSENT_ALIAS_MODE" description="PLG_SYSTEM_DIRECT_ALIAS_ABSENT_TIP_DESC">
								<option value="1">PLG_SYSTEM_DIRECT_ALIAS_ABSENT</option>
								<option value="0">PLG_SYSTEM_DIRECT_ALIAS_PRESENT</option>
							</field>
						</fieldset>
					</fields>
				</form>',
                false
            );
        }
    }

    /**
     * @since        1.0
     * @noinspection PhpUnused
     */
    public function onAfterInitialise(): void
    {
        // Falang overloads menu items via own router's parse rule, so we need to update routes after its rule but not now.
        if (static::$has_falang) {
            $this->getRouter()->attachParseRule([$this, 'updateDirectRoutes'], Router::PROCESS_BEFORE);
        } else {
            $this->updateDirectRoutes();
        }
    }

    /**
     * Joomla4 requires parent_id=1 for top-level shorten URLs.
     * @since 1.0
     */
    public function updateDirectRoutes(): void
    {
        // Execute only once since method can be attached as parse rule and executed multiple times.
        static $updated = 0;
        if ($updated++) {
            return;
        }

        // Just shorten all URLs.
        if ($this->params->get('shorten_all')) {
            foreach ($this->getMenu()->getMenu() as $item) {
                $item->getParams()->set('_route', $item->route);
                $item->route = $item->alias;
            }
        } // Or custom settings per menu item
        else {
            $parent_routes = [];

            foreach ($this->getMenu()->getMenu() as $item) {
                $params = $item->getParams();

                // Apply direct alias
                if ($params->get('direct_alias')) {
                    $params->set('_route', $item->route);
                    $item->route                                        = $item->alias;
                    $this->custom_routes[$item->language][$item->route] = $item->id;
                    $parent_routes[$item->id]                           = $item->route;
                } // Apply changed parent route
                elseif (isset($parent_routes[$item->parent_id])) {
                    $params->set('_route', $item->route);
                    $item->route                                        = \ltrim(
                        $parent_routes[$item->parent_id] . '/' . $item->alias,
                        '/'
                    );
                    $this->custom_routes[$item->language][$item->route] = $item->id;
                    $parent_routes[$item->id]                           = $item->route;
                }

                // Remember changed parent route
                if ($params->get('absent_alias')) {
                    $parent_routes[$item->id] = \trim(\dirname($item->route), './');
                }
            }
        }

        // Decorate router
        if ($this->custom_routes || $this->params->get('shorten_all')) {
            $getDuringRules = (static function &($router) {
                return $router->rules['parse' . Router::PROCESS_DURING];
            })->bindTo(null, SiteRouter::class);

            foreach ($getDuringRules($this->getRouter()) as &$callback) {
                if (\is_array($callback)
                    && $callback[1] === 'parseSefRoute'
                    && \is_object($callback[0])
                    && \get_class($callback[0]) === SiteRouter::class
                ) {
                    $callback = [$this, 'parseSefRoute'];
                    break;
                }
            }
        }
    }

    /**
     * @param  SiteRouter  $router  Router object
     * @param  Uri         $uri     URI object to process
     *
     * @since 3.0.0
     * @see   SiteRouter::parseSefRoute()
     */
    public function parseSefRoute($router, $uri): void
    {
        $route = $uri->getPath();

        // If the URL is empty, we handle this in the non-SEF parse URL
        if (empty($route)) {
            return;
        }

        // Parse the application route
        $segments = \explode('/', $route);

// CUSTOM START
        /** @var SiteApplication $app */
        $app = $this->getApplication();

        $menu = $this->getMenu();
// CUSTOM END

        if (\count($segments) > 1 && $segments[0] === 'component') {
            $uri->setVar('option', 'com_' . $segments[1]);
            $uri->setVar('Itemid', null);
            $route = \implode('/', \array_slice($segments, 2));
        } else {
// CUSTOM START
            $lang_tag = $app->getLanguage()->getTag();
            $found    = null;

            if ($this->params->get('shorten_all')) {
                // All short
                foreach ($menu->getMenu() as $item) {
                    if ($item->alias === $segments[0]
                        && (!$app->getLanguageFilter() || ($item->language === '*' || $item->language === $lang_tag))
                    ) {
                        $found = $item;
                        break;
                    }
                }
            } elseif ($this->custom_routes) {
                $segmentsTest = \explode('/', $route);
                // Custom short
                while ($segmentsTest) {
                    $routeTest = \implode('/', $segmentsTest);

                    if (isset($this->custom_routes[$lang_tag][$routeTest])) {
                        $found = $menu->getItem($this->custom_routes[$lang_tag][$routeTest]);
                        break;
                    }

                    if (isset($this->custom_routes['*'][$routeTest])) {
                        $found = $menu->getItem($this->custom_routes['*'][$routeTest]);
                        break;
                    }

                    \array_pop($segmentsTest);
                }
            }

            // Original usual logic
            if (!$found) {
                /** @noinspection PhpParamsInspection */
                $items = $menu->getItems(['parent_id', 'access'], [1, null]);

                foreach ($segments as $segment) {
                    $matched = false;

                    foreach ($items as $item) {
                        if ($item->alias === $segment
                            && (!$app->getLanguageFilter() || ($item->language === '*' || $item->language === $lang_tag))
                        ) {
                            $found   = $item;
                            $matched = true;
                            $items   = $item->getChildren();
                            break;
                        }
                    }

                    if (!$matched) {
                        break;
                    }
                }
            }
// CUSTOM END

//            // Get menu items.
//            $items    = $menu->getItems(['parent_id', 'access'], [1, null]);
//            $lang_tag = $this->app->getLanguage()->getTag();
//            $found    = null;
//
//            foreach ($segments as $segment) {
//                $matched = false;
//
//                foreach ($items as $item) {
//                    if (
//                        $item->alias == $segment
//                        && (!$this->app->getLanguageFilter()
//                            || ($item->language === '*'
//                                || $item->language === $lang_tag))
//                    ) {
//                        $found   = $item;
//                        $matched = true;
//                        $items   = $item->getChildren();
//                        break;
//                    }
//                }
//
//                if (!$matched) {
//                    break;
//                }
//            }

            // Menu links are not valid URLs. Find the first parent that isn't a menulink
            if ($found && $found->type === 'menulink') {
                while ($found->hasParent() && $found->type === 'menulink') {
                    $found = $found->getParent();
                }

                if ($found->type === 'menulink') {
                    $found = null;
                }
            }

            if (!$found) {
                $found = $menu->getDefault($lang_tag);
            } else {
                $route = \trim(\substr($route, \strlen($found->route)), '/');
            }

            if ($found) {
                if ($found->type === 'alias') {
                    $newItem = $menu->getItem($found->getParams()->get('aliasoptions'));

                    if ($newItem) {
                        $found->query     = \array_merge($found->query, $newItem->query);
                        $found->component = $newItem->component;
                    }
                }

                $uri->setVar('Itemid', $found->id);
                $uri->setVar('option', $found->component);
            }
        }

        // Set the active menu item
        if ($uri->getVar('Itemid')) {
            $menu->setActive($uri->getVar('Itemid'));
        }

        // Parse the component route
        if (!empty($route) && $uri->getVar('option')) {
            $segments = \explode('/', $route);

            if (\count($segments)) {
                // Handle component route
                $component = \preg_replace('/[^A-Z0-9_.-]/i', '', $uri->getVar('option'));
                $cRouter   = $router->getComponentRouter($component);
                $uri->setQuery(\array_merge($uri->getQuery(true), $cRouter->parse($segments)));
            }

            $route = \implode('/', $segments);
        }

        $uri->setPath($route);
    }

    /**
     * @since 3.1.1+
     */
    protected function getRouter(): SiteRouter
    {
        return $this->router ?? ($this->router = Factory::getContainer()->get(SiteRouter::class));
    }

    /**
     * @since 3.1.1+
     */
    protected function getMenu(): SiteMenu
    {
        if ($this->getApplication()->isClient('site')) {
            return $this->getApplication()->getMenu('site');
        }

        // SiteRouter in AdministratorApplication actually uses menu from SiteApplication
        return Factory::getContainer()->get(SiteApplication::class)->getMenu('site');
    }
}
