<?php
/**
 * @package        Alter Fields
 * @copyright      Copyright (C) 2022-2023 AlterBrains.com. All rights reserved.
 * @license        https://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL
 */

/** @noinspection PhpMultipleClassDeclarationsInspection */

namespace AlterBrains\Plugin\System\Alterfields\Extension;

use Joomla\CMS\Application\AdministratorApplication;
use Joomla\CMS\Application\SiteApplication;
use Joomla\CMS\Document\HtmlDocument;
use Joomla\CMS\Event\Model;
use Joomla\CMS\Factory;
use Joomla\CMS\Fields\FieldsServiceInterface;
use Joomla\CMS\Form\Form;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Layout\FileLayout;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Profiler\Profiler;
use Joomla\CMS\Uri\Uri;
use Joomla\Component\Fields\Administrator\Helper\FieldsHelper;
use Joomla\Component\Fields\Administrator\Model\FieldsModel;
use Joomla\Database\DatabaseAwareInterface;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Event\Event;
use Joomla\Event\SubscriberInterface;
use Joomla\Filesystem\Path;
use Joomla\Registry\Registry;

\defined('_JEXEC') or die;

/**
 * @since        1.0
 * @noinspection PhpUnused
 */
class Alterfields extends CMSPlugin implements SubscriberInterface, DatabaseAwareInterface
{
    use DatabaseAwareTrait;

    /**
     * @var SiteApplication|AdministratorApplication
     * @since        1.0
     * @noinspection PhpMissingFieldTypeInspection
     */
    private $application;

    /**
     * @since 1.0
     */
    protected static string|bool|null $context = null;

    /**
     * @since 1.0
     */
    protected static string $contextMap;

    /**
     * @var \stdClass[]
     * @since 1.0
     */
    protected static ?array $fields = null;

    /**
     * All fields, required to get subform children which can be missed in $fields.
     * @var \stdClass[]
     * @since 1.2.0
     */
    protected static ?array $fieldsAll = null;

    /**
     * @since 1.0
     */
    public static ?Registry $listModelState = null;

    /**
     * @inheritDoc
     * @since 1.0
     */
    public function __construct($dispatcher, array $config = [])
    {
        parent::__construct($dispatcher, $config);

        $this->application = Factory::getApplication();

        self::$contextMap = $this->params->get('contextMap', '');
    }

    /**
     * @inheritDoc
     * @since 1.0
     */
    public static function getSubscribedEvents(): array
    {
        return [
            'onContentPrepareForm' => 'onContentPrepareForm',
            'onAfterDispatch'      => 'onAfterDispatch',
        ];
    }

    /**
     * @since  1.0
     */
    public function onContentPrepareForm(Event/*Model\PrepareFormEvent */ $event): void
    {
        /**
         * @var Form  $form
         * @var mixed $data
         */
        [$form, $data] = $event instanceof Model\PrepareFormEvent
            ? [$event->getForm(), $event->getData()]
            : $event->getArguments();

        switch ($form->getName()) {
            case 'com_fields.field.' . $this->application->input->get('context'):
                if (!empty($data->type)) {
                    $this->loadLanguage();
                    $form->loadFile(__DIR__ . '/../../forms/com_fields.field.xml', false);
                    // todo, 'repeatable' is Joomla3 EOL
                    if ($data->type === 'subform' || $data->type === 'repeatable') {
                        $form->removeField('af_orderable', 'params');
                        if (!empty($data->params['af_orderable'])) {
                            $data->params['af_orderable'] = '0';
                        }
                    }
                }
                break;

            case 'com_fields.fields.filter':
                if ($this->params->get('backend_fields_info')) {
                    $this->decorateFieldList();
                }
                break;
        }
    }

    /**
     * @since 1.0
     */
    public function onAfterDispatch(): void
    {
        JDEBUG and Profiler::getInstance('Application')->mark('before Alter Fields list');

        $this->decorateList();

        JDEBUG and Profiler::getInstance('Application')->mark('after Alter Fields list');
    }

    /**
     * @since 1.0
     */
    protected function decorateList(): void
    {
        // Get fields to display in columns
        $fields = static::getFields(static function ($field) {
            return $field->af_listable && empty($field->only_use_in_subform);
        });
        if (!$fields) {
            return;
        }

        if (\function_exists('ini_set') && ($limit = $this->params->get('pcre_backtrack_limit'))) {
            \ini_set('pcre_backtrack_limit', $limit);
        }

        /** @var HtmlDocument $document */
        $document = $this->application->getDocument();

        $document->setBuffer(
            \preg_replace_callback(
            //'~<table(.+?)name="checkall-toggle"(.+?)</table>~s',
                '~<table\b[^>]*>.*?name="checkall-toggle".*?</table>~s',
                function ($matches) use ($fields) {
                    return $this->decorateListTable($matches[0], $fields);
                },
                $document->getBuffer('component') ?? '',
                1
            ),
            'component'
        );

        // Show PCRE error
        if (($document->getBuffer('component') ?? '') === '' && (0 !== $errorCode = \preg_last_error())) {
            $errorText = $errorCode === \PREG_BACKTRACK_LIMIT_ERROR
                ? 'PLG_SYSTEM_ALTERFIELDS_ERROR_PCRE_BACKTRACK_LIMIT_EXHAUSTED'
                : 'PLG_SYSTEM_ALTERFIELDS_ERROR_PCRE';
            $this->loadLanguage();
            $this->application->enqueueMessage(
                Text::sprintf($errorText, \preg_last_error_msg(), \ini_get('pcre.backtrack_limit')),
                'error'
            );
        }
    }

    /**
     * @param  \stdClass[]  $fields
     *
     * @since 1.0
     */
    protected function decorateListTable(string $html, array $fields): string
    {
        // Extract thead.
        $html = \preg_replace_callback(
        //'~<thead(.+?)</thead>~us',
            '~<thead\b[^>]*>.*?</thead>~s',
            function ($matches) use ($fields) {
                return $this->decorateListTableHead($matches[0], $fields);
            },
            $html,
            1
        );

        // Extract tbody.
        return \preg_replace_callback(
        //'~<tbody(.+?)</tbody>~us',
            '~<tbody\b[^>]*>.*?</tbody>~s',
            function ($matches) use ($fields) {
                return $this->decorateListTableBody($matches[0], $fields);
            },
            $html,
            1
        );
    }

    /**
     * @param  \stdClass[]  $fields
     *
     * @since 1.0
     */
    protected function decorateListTableHead(string $html, array $fields): string
    {
        if (self::$listModelState) {
            $listOrder = \htmlspecialchars(self::$listModelState->get('list.ordering', ''));
            $listDirn  = \htmlspecialchars(self::$listModelState->get('list.direction', ''));
        } else {
            $listOrder = '';
            $listDirn  = '';
        }

        // Titles, note: can be <th> or <td>
        $titlesHtml = [];
        foreach ($fields as $field) {
            if ($field->af_orderable) {
                // Field key in request.
                $field->af_name = 'af_' . $field->id;

                $title = HTMLHelper::_(
                    'searchtools.sort',
                    $field->af_list_title,
                    $field->af_name,
                    $listDirn,
                    $listOrder
                );
            } else {
                $title = \htmlspecialchars($field->af_list_title);
            }

            $class        = $field->params->get('af_th_class');
            $titlesHtml[] = '<th' . ($class ? ' class="' . $class . '"' : '') . '>' . $title . ' </th>';
        }

        return \strtr($html, [
            '</tr>' => \implode("\n", $titlesHtml) . '</tr>',
        ]);
    }

    /**
     * @param  \stdClass[]  $fields
     *
     * @since 1.0
     */
    protected function decorateListTableBody(string $html, array $fields): string
    {
        $tbody = $html;

        // Values
        //\preg_match_all('~<td(.+?)name="cid\[]" value="(\d+)"(.+?)</tr>~s', $tbody, $matches);
        \preg_match_all('~<td(.*?)\bname="cid\[]"\s+value="(\d+)"(.*?)</tr>~s', $tbody, $matches);

        if (!isset($matches[0][0])) {
            return $tbody;
        }

        $replacements = [];

        $db = $this->getDatabase();

        $ids = \array_map([$db, 'quote'], $matches[2]);

        $query = 'SELECT * 
				FROM #__fields_values 
				WHERE field_id IN(' . \implode(',', \array_keys($fields)) . ') 
				    AND item_id IN(' . \implode(',', $ids) . ')';
        $db->setQuery($query);
        $allValues = $db->loadObjectList();

        $itemValues = [];

        foreach ($allValues as $value) {
            // Multiple? Transform to array.
            if (isset($itemValues[$value->item_id][$value->field_id])) {
                if (!\is_array($itemValues[$value->item_id][$value->field_id])) {
                    $itemValues[$value->item_id][$value->field_id] = [$itemValues[$value->item_id][$value->field_id]];
                }

                $itemValues[$value->item_id][$value->field_id][] = $value->value;

                continue;
            }

            // String by default
            $itemValues[$value->item_id][$value->field_id] = $value->value;
        }

        PluginHelper::importPlugin('fields');

        foreach ($matches[0] as $i => $full) {
            $fieldsHtml = [];

            foreach ($fields as $field) {
                $fieldHtml = '';

                // Set default value
                if (!isset($itemValues[$matches[2][$i]][$field->id])
                    && ($field->default_value ?? '') !== ''
                    && $field->params->get('af_list_default')
                ) {
                    $itemValues[$matches[2][$i]][$field->id] = $field->default_value;
                }

                if (isset($itemValues[$matches[2][$i]][$field->id])) {
                    $field->value    = $itemValues[$matches[2][$i]][$field->id];
                    $field->rawvalue = $field->value;

                    // Get rendered output, see FieldsHelper::getFields()

                    $item = new \stdClass();

                    // Event allow plugins to modify the output of the field before it is prepared
                    /** @noinspection PhpDeprecationInspection */
                    $this->application->triggerEvent(
                        'onCustomFieldsBeforePrepareField',
                        [self::$context, $item, &$field]
                    );

                    // Gathering the value for the field
                    /** @noinspection PhpDeprecationInspection */
                    $value = $this->application->triggerEvent(
                        'onCustomFieldsPrepareField',
                        [self::$context, $item, &$field]
                    );

                    if (\is_array($value)) {
                        $value = \implode(' ', $value);
                    }

                    // Event allow plugins to modify the output of the prepared field
                    /** @noinspection PhpDeprecationInspection */
                    $this->application->triggerEvent(
                        'onCustomFieldsAfterPrepareField',
                        [self::$context, $item, $field, &$value]
                    );

                    // Skip rendering empty values.
                    if ($value !== '' && $value !== []) {
                        // Assign the value
                        $field->value = $value;

                        // Some custom preparations
                        $this->prepareFieldBeforeListRender($field);

                        // Use plain output.
                        //$fieldHtml = $value;

                        // Note: frontend template paths are not used! Only backend template paths.
                        $fieldHtml = $this->renderField(
                            self::$context,
                            'field.' . $field->params->get('af_layout', 'render'),
                            [
                                'item'    => $item,
                                'context' => self::$context,
                                'field'   => $field,
                            ]
                        );
                    }
                }

                $class = $field->params->get('af_td_class');
                $style = $field->params->get('af_td_style');
                if ($data = $field->params->get('af_td_data_value')) {
                    $dataValue = \is_array($field->rawvalue)
                        ? \json_encode($field->rawvalue, \JSON_THROW_ON_ERROR)
                        : $field->rawvalue;
                }

                /** @noinspection PhpUndefinedVariableInspection */
                $fieldHtml = '<td'
                    . ($data ? ' data-value="' . \htmlspecialchars($dataValue, \ENT_COMPAT) . '"' : '')
                    . ($class ? ' class="' . $class . '"' : '')
                    . ($style ? ' style="' . $style . '"' : '') . '>' . $fieldHtml . '</td>';

                $fieldsHtml[] = $fieldHtml;
            }

            $replacements[$full] = \strtr($full, [
                '</tr>' => \implode("\n", $fieldsHtml) . '</tr>',
            ]);
        }

        return \strtr($tbody, $replacements);
    }

    /**
     * @since 1.0
     */
    protected function prepareFieldBeforeListRender(\stdClass $field): void
    {
        /** @noinspection DegradedSwitchInspection */
        /** @noinspection PhpSwitchStatementWitSingleBranchInspection */
        switch ($field->type) {
            case 'media':
                $field->value = \strtr($field->value, [
                    ' src="' => ' style="max-width:100%;height:auto" src="' . Uri::root(true) . '/',
                ]);
                break;
        }
    }

    /**
     * @param  ?string  $realModel  Can be case-sensitive, i.e. AdminEvents
     *
     * @return string|false FALSE if undetected
     * @since 1.0
     */
    public static function getContext(
        ?string &$realOption = null,
        ?string &$realView = null,
        ?string &$realModel = null
    ): string|bool {
        static $option, $view, $model;

        if (self::$context === null) {
            $app = Factory::getApplication();

            $map = static::getContextMap();

            $option = $app->input->get('option');

            // Note default empty string required for empty view
            $view = $app->input->get('view', '');

            // Special case for com_categories
            if ($option === 'com_categories') {
                $view      = 'categories';
                $extension = $app->input->get('extension', $map['com_categories']['_default'] ?? 'com_content');

                self::$context = $map[$option][$extension] ?? false;
            } else {
                self::$context = $map[$option][$view] ?? false;

                // Default view
                if ($view === '' && isset($map[$option][''])) {
                    $view = $map[$option]['_default'];
                }
            }

            $model = $map[$option]['_models'][$view] ?? $view;
        }

        $realOption = $option;
        $realView   = $view;
        $realModel  = $model;

        return self::$context;
    }

    /**
     * @since 1.0
     */
    protected static function getContextMap(): array
    {
        $contextMapData = \explode(
            "\n",
            \strtr(\trim(static::$contextMap), [
                "\r\n" => "\n",
                ' '    => '',
            ])
        );

        $contextMap = [];

        // Prepare map, each line is "fieldsContext|com_option:view1[?],[$view2]"
        foreach ($contextMapData as $map1) {
            if (('' === $map1 = \trim($map1)) || !\str_contains($map1, '|')) {
                continue;
            }

            [$context, $componentView] = @\explode('|', $map1, 2);

            if (\str_contains($componentView, ':')) {
                [$component, $views] = \explode(':', $componentView, 2);

                foreach (\explode(',', $views) as $view) {
                    // Custom model name for view
                    if (\str_contains($view, '>')) {
                        [$view, $model] = \explode('>', $view, 2);

                        $contextMap[$component]['_models'][$view] = $model;
                    }

                    // Default view
                    if (\str_ends_with($view, '?')) {
                        $view = \substr($view, 0, -1);

                        $contextMap[$component]['']         = $context;
                        $contextMap[$component]['_default'] = $view;
                    }

                    $contextMap[$component][$view] = $context;
                }
            }
        }

        return $contextMap;
    }

    /**
     * @return \stdClass[]
     * @since 1.0
     */
    public static function &getFields(callable $filterCallback = null): array
    {
        // Load all fields.
        if (self::$fieldsAll === null) {
            self::$fieldsAll = [];

            // Get all custom field instances
            if (false !== $context = static::getContext()) {
                foreach (FieldsHelper::getFields($context, null, false, null, true) as $customField) {
                    static::$fieldsAll[$customField->id] = $customField;
                }
            }
        }

        if (self::$fields === null) {
            self::$fields = [];

            if (false !== $context = static::getContext($option, $view)) {
                // Detect item category
                if ($filterCategoryId = Factory::getApplication()->getUserState(
                    $option . '.' . $view . '.filter.category_id'
                )) {
                    $item = (object)['fieldscatid' => $filterCategoryId];
                }

                // All fields, including subform-only fields.
                $fields = [];
                foreach (FieldsHelper::getFields($context, $item ?? null, false, [], true) as $field) {
                    $fields[$field->id] = $field;
                }

                // Subform-only fields can be missed if we have selected category, inject from allFields cache.
                foreach ($fields as $field) {
                    // Subform contains children in options.
                    if ($field->type === 'subform') {
                        foreach ($field->fieldparams->get('options', []) as $opt) {
                            if (!isset($fields[$opt->customfield])) {
                                if (!isset(self::$fieldsAll[$opt->customfield])) {
                                    continue;
                                }
                                $fields[$opt->customfield] = self::$fieldsAll[$opt->customfield];
                            }
                        }
                    }
                }

                foreach ($fields as $field) {
                    self::$fields[$field->id] = $field;

                    // Hide display label
                    $field->params->set('showlabel', 0);

                    $field->af_list_title = Text::_(
                        $field->af_list_title ?? $field->params->get('af_list_title', $field->title)
                    );

                    $field->af_listable   = $field->params->get('af_listable');
                    $field->af_orderable  = $field->params->get('af_orderable');
                    $field->af_searchable = $field->params->get('af_searchable');
                    $field->af_filterable = $field->params->get('af_filterable');

                    // subform and children are not orderable. todo - repeatable is Joomla3 EOL
                    if (!empty($field->only_use_in_subform)
                        || $field->type === 'subform'
                        || $field->type === 'repeatable'
                    ) {
                        $field->af_orderable = false;
                    }
                }
            }
        }

        if (!$filterCallback) {
            return self::$fields;
        }

        $fields = [];
        foreach (self::$fields as $field) {
            if ($filterCallback($field)) {
                $fields[$field->id] = $field;
            }
        }

        return $fields;
    }

    /**
     * @since 1.2.0
     */
    protected function decorateFieldList(): void
    {
        $debug     = \debug_backtrace(\DEBUG_BACKTRACE_PROVIDE_OBJECT, 4);
        $classname = FieldsModel::class;
        if (!empty($debug[3]['object']) && $debug[3]['object'] instanceof $classname) {
            foreach ($debug[3]['object']->getItems() as $item) {
                $badges = \array_filter([
                    $item->params->get('af_listable') ? Text::_(
                        'PLG_SYSTEM_ALTERFIELDS_BACKEND_FIELDS_INFO_LISTABLE'
                    ) : '',
                    $item->params->get('af_orderable') ? Text::_(
                        'PLG_SYSTEM_ALTERFIELDS_BACKEND_FIELDS_INFO_ORDERABLE'
                    ) : '',
                    $item->params->get('af_filterable') ? Text::sprintf(
                        'PLG_SYSTEM_ALTERFIELDS_BACKEND_FIELDS_INFO_FILTERABLE',
                        $this->getSearchModeTitle($item->params->get('af_filter_mode'))
                    ) : '',
                    $item->params->get('af_searchable') ? Text::sprintf(
                        'PLG_SYSTEM_ALTERFIELDS_BACKEND_FIELDS_INFO_SEARCHABLE',
                        $this->getSearchModeTitle($item->params->get('af_search_mode'))
                    ) : '',
                ]);
                if ($badges) {
                    $item->note = $item->note ? $item->note . '. ' : '';
                    $item->note .= \implode(', ', $badges);
                }
            }
        }
    }

    /**
     * @since 1.2.0
     */
    protected function getSearchModeTitle(?string $mode): string
    {
        return $mode ? Text::_('PLG_SYSTEM_ALTERFIELDS_FIND_MODE_' . $mode) : 'n/a';
    }

    /**
     * Duplicates original method but prepends frontend template paths missed in original code.
     * @see   FieldsHelper::render
     * @since 2.1.1+
     */
    protected function renderField(string $context, string $layoutFile, array $displayData): ?string
    {
        $value = '';

        $prependsPaths = [];

        /*
         * Because the layout refreshes the paths before the render function is
         * called, so there is no way to load the layout overrides in the order
         * template -> context -> fields.
         * If there is no override in the context then we need to call the
         * layout from Fields.
         */
        if ($parts = self::extractFieldContext($context)) {
            // Trying to render the layout on the component from the context
            $value = $this->renderFieldLayout(
                $layoutFile,
                $displayData,
                null,
                $prependsPaths + ['component' => $parts[0], 'client' => 0]
            );
        }

        if (($value ?? '') === '') {
            // Trying to render the layout on Fields itself
            $value = $this->renderFieldLayout(
                $layoutFile,
                $displayData,
                null,
                $prependsPaths + ['component' => 'com_fields', 'client' => 0]
            );
        }

        return $value;
    }

    /**
     * @since 2.1.1+
     */
    protected function renderFieldLayout(
        string $layoutFile,
        ?array $displayData,
        ?string $basePath,
        array $options
    ): ?string {
        static $includePaths = null;

        if ($includePaths === null) {
            $db    = $this->getDatabase();
            $query = $db->getQuery(true);

            // Build the query.
            $query->select('element, name')
                ->from('#__extensions')
                ->where($db->quoteName('client_id') . ' = 0')
                ->where($db->quoteName('type') . ' = ' . $db->quote('template'))
                ->where($db->quoteName('enabled') . ' = 1');

            // Set the query and load the templates.
            $db->setQuery($query);
            $templates = $db->loadObjectList('element');

            $extension = $this->application->getInput()->get('option');

            $includePaths = [];
            foreach ($templates as $template) {
                $includePaths = \array_merge($includePaths, [
                    Path::clean(JPATH_SITE . '/templates/' . $template->element . '/html/layouts/' . $extension),
                    Path::clean(JPATH_SITE . '/templates/' . $template->element . '/html/layouts/com_fields'),
                    Path::clean(JPATH_SITE . '/templates/' . $template->element . '/html/layouts'),
                ]);
            }
        }

        $layout = new FileLayout($layoutFile, $basePath, $options);

        $layout->addIncludePaths($includePaths);

        return $layout->render($displayData);
    }

    /**
     * Copy from Joomla code.
     * @see     FieldsHelper::extract
     * @since   2.1.1
     */
    public static function extractFieldContext(?string $contextString, ?object $item = null): ?array
    {
        if ($contextString === null) {
            return null;
        }

        $parts = \explode('.', $contextString, 2);

        if (\count($parts) < 2) {
            return null;
        }

        $newSection = '';

        $component = Factory::getApplication()->bootComponent($parts[0]);

        if ($component instanceof FieldsServiceInterface) {
            $newSection = $component->validateSection($parts[1], $item);
        }

        if ($newSection) {
            $parts[1] = $newSection;
        }

        return $parts;
    }
}
