/**
 * @package      AJAX Toggler
 * @copyright    Copyright (C) 2009 - 2022 AlterBrains.com. All rights reserved.
 * @license      Commercial
 */
(function ($) {
	'use strict';

	// noinspection JSUnusedLocalSymbols,JSUnresolvedVariable
	window.AjaxToggler = {
		options: {
			base: null,
			menu: true,
			toolbarTasks: ['publish', 'unpublish', 'featured', 'unfeatured', 'archive', 'checkin', 'trash', 'delete', 'rebuild', 'duplicate', 'refresh', 'block', 'unblock', 'activate'],
			edit: true
		},

		form: null,
		table: null,
		wrapper: null,
		overlay: null,

		current_task: null,
		flag: null,
		flag_ordering: null,
		Joomla4: null,
		jsTools: null,

		initialize: function (_options) {
			$.extend(this.options, _options);

			// noinspection JSUnresolvedVariable
			if (window.Joomla && document.adminForm && this.parseTableForm(document.adminForm)) {
				this.Joomla4 = !Joomla.loadingLayer;
				this.parseJoomla();
			}
		},

		parseTableForm: function (form) {
			// noinspection JSUnresolvedVariable
			if (!form || !form.task || !form.boxchecked) {
				return false;
			}

			this.form = form;

			//this.table = $(form).find('table.table');
			// Joomla3 can have inner div.
			this.table = $('#j-main-container > table.table, #j-main-container > div > table.table').first();

			if (!this.table.length) {
				return false;
			}

			this.current_task = form.task.value;

			// create flag
			if (typeof (this.form['jatoggler']) === 'undefined') {
				this.flag = $('<input type="hidden" name="jatoggler" value="0" />').appendTo(this.form);
			}
			if (typeof (this.form['jatoggler_ordering']) === 'undefined') {
				this.flag_ordering = $('<input type="hidden" name="jatoggler_ordering" value="" />').appendTo(this.form);
			}

			return true;
		},

		parseJoomla: function () {
			if (!window.Joomla._ajaxtoggler) {
				window.Joomla._ajaxtoggler = true;

				// Custom submitform
				window.Joomla._submitform = window.Joomla.submitform;
				window.Joomla.submitform = function (task, form, validate) {
					this.submitTableForm(task, form, validate);
				}.bind(this);

				// Custom tableOrdering
				window.Joomla._tableOrdering = window.Joomla.tableOrdering;
				window.Joomla.tableOrdering = function (order, dir, task, form) {
					this.tableOrdering(order, dir, task, form);
				}.bind(this);

				// Custom listItemTask
				window.Joomla._listItemTask = window.Joomla.listItemTask;
				window.Joomla.listItemTask = function (id, task, form = null) {
					return this.listItemTask(id, task, form);
				}.bind(this);
			}

			// Joomla 3.2.1+, 4.x Table ordering
			this.jsTools = $('.js-stools');

			if (this.jsTools.length) {
				// Capture click on TH to prevent any Joomla events on a.click
				document.querySelectorAll('a.js-stools-column-order').forEach(function (a) {
					a.parentNode.addEventListener('click', function (e) {
						e.preventDefault();
						e.stopImmediatePropagation();

						var target = $(e.target);

						// can be executed for inner element, so check for A
						if (e.target.tagName !== 'A') {
							target = target.closest('th').find('a');
						}

						if (target.length) {
							var order = target.attr('data-order');
							var dir = target.attr('data-direction');

							$('#list_fullordering').val(order + ' ' + dir).trigger('liszt:updated');

							this.tableOrdering(order, dir, '', null);
						}
					}.bind(this), true);
				}.bind(this));
			}
		},

		createWrapper: function () {
			if (!this.wrapper) {
				this.table.wrap('<div id="ajaxtogglerWrapper" style="clear:both"></div>');
				// Note that div#ajaxtogglerWrapper can be loaded if no results was filtered
				this.wrapper = $('#ajaxtogglerWrapper');
			}
		},

		/**
		 * @param {node} parent
		 */
		createOverlay: function (parent) {
			document.body.style.cursor = 'wait';

			// Custom case, we need to adopt native Joomla layer a bit
			if (parent) {
				$(parent).css({position: 'relative'});

				var layer;

				// Joomla3 compat
				if (Joomla.loadingLayer) {
					layer = Joomla.loadingLayer('show', $(parent).get(0));
					$(layer).css({position: 'absolute', bottom: '0'});
				} else {
					layer = document.createElement('joomla-core-loader');
					$(parent).get(0).prepend(layer);
					$(layer).css({position: 'absolute', bottom: '0'});
				}

				// prettify wrapper
				if (parent === this.wrapper) {
					this.wrapper.css({overflow: 'hidden', zIndex: '1', position: 'relative'});
				}
			}
			// Simple case
			else {
				// Joomla3 compat
				if (Joomla.loadingLayer) {
					Joomla.loadingLayer('show');
				}
			}
		},

		removeOverlay: function () {
			document.body.style.cursor = '';

			var layer = '';

			// Joomla3 compat
			if (Joomla.loadingLayer) {
				layer = Joomla.loadingLayer('hide');
				layer.parentNode.removeChild(layer);
			}

			// Remove wrapper style if any.
			if (this.wrapper) {
				this.wrapper.css({overflow: '', zIndex: '', position: ''});
			}
		},

		/**
		 * Generic submit form
		 *
		 * @param  {String}  task      The given task
		 * @param  {node}    form      The form element
		 * @param  {bool}    validate  The form element
		 *
		 * @returns  {boolean} Note: we need to return Bool here, not like in Joomla!
		 */
		submitTableForm: function (task, form, validate) {
			// usual submit if we have task
			if (typeof (task) !== 'undefined' && task !== '') {

				var realTask = task;
				if (realTask.indexOf('.') !== -1) {
					realTask = realTask.split('.')[1];
					if ($.inArray(realTask, this.options.toolbarTasks) > -1) {
						this.form.task.value = task;
						this.loadTable();
						return false;
					}
				}

				this.form.task.value = task;

				// Submit the form.
				if (typeof this.form.onsubmit === 'function') {
					this.form.onsubmit();
				}
				if (typeof this.form.fireEvent === 'function') {
					this.form.fireEvent('submit');
				}
				this.form.submit();
				return false;
			}

			this.loadTable();

			return false;
		},

		/**
		 * USED IN: libraries/joomla/html/html/grid.php
		 * In other words, on any reorderable table
		 *
		 * @param  {string}  order  The order value
		 * @param  {string}  dir    The direction
		 * @param  {string}  task   The task
		 * @param  {node}    form   The form
		 *
		 * @returns  {boolean} Note: we need to return Bool here, not like in Joomla!
		 */
		tableOrdering: function (order, dir, task, form) {
			if (typeof (form) === 'undefined' || form === null) {
				form = this.form;
			}

			// noinspection JSUnresolvedVariable
			if (form.filter_order !== undefined) {
				// noinspection JSUnresolvedVariable
				this.form.filter_order.value = order;
			}
			// noinspection JSUnresolvedVariable
			if (form.filter_order_Dir !== undefined) {
				// noinspection JSUnresolvedVariable
				this.form.filter_order_Dir.value = dir;
			}

			$('#sortTable').val(order).trigger('liszt:updated');
			$('#directionTable').val(dir).trigger('liszt:updated');

			this.form.task.value = task;

			this.loadTable();

			return false;
		},

		/**
		 * USED IN: all over :)
		 *
		 * @param  {string}  id    The id
		 * @param  {string}  task  The task
		 * @param  {HTMLFormElement}  form  The form
		 *
		 * @return {boolean}
		 */
		listItemTask: function (id, task, form = null) {
			if (form === null) {
				form = this.form;
			}
			const cb = form[id];
			if (cb) {
				for (let i = 0; true; i++) {
					let cbx = form['cb' + i];
					if (!cbx)
						break;
					cbx.checked = false;
				} // for
				cb.checked = true;
				// noinspection JSUnresolvedVariable
				form.boxchecked.value = 1;

				form.task.value = task;
				this.loadTable();
			}

			return false;
		},

		loadTable: function () {
			this.createWrapper();
			this.createOverlay(this.wrapper);
			this.flag.val(1);

			// Is ordering loaded?
			if (this.Joomla4) {
				// noinspection JSUnresolvedVariable
				this.flag_ordering.val(window.dragula ? '1' : '');
			} else {
				this.flag_ordering.val(jQuery.JSortableList ? '1' : '');
			}

			// Disable activated list-required buttons
			document.querySelectorAll('joomla-toolbar-button[list-selection]').forEach(el => {
				el.setDisabled(true);
			});

			$.ajax({
					type: 'POST',
					url: this.form.action,
					dataType: 'json',
					cache: false,
					data: $(this.form).serialize(),
					context: this
				}
			).done(function (resp) {
					// Main table
					$('#ajaxtogglerWrapper').html(resp.data.html);

					// Toolbar
					if (resp.data.toolbar) {
						if (this.Joomla4) {
							$('#toolbar').parent().replaceWith(resp.data.toolbar);
						} else {
							$('#toolbar').replaceWith(resp.data.toolbar);
						}
					}

					// Pagination
					$('.pagination.pagination-toolbar').html(resp.data.pagination);

					// Messages
					//Joomla.removeMessages();
					// Clear messages immediately, otherwise Joomla4 fades them first, and we see a blood from our eyes
					document.getElementById('system-message-container').innerHTML = '';

					if (resp.messages) {
						Joomla.renderMessages(resp.messages);
					}

					this.flag.val(0);
					this.form.task.value = this.current_task;
					// noinspection JSUnresolvedVariable
					this.form.boxchecked.value = '';

					this.removeOverlay();

					this.parseJoomla();
					this.checkScroll();

					this.checkContext();
					this.checkTooltips();
					this.checkPopovers();
					this.checkMultiselect();
					this.checkScripts();
				}.bind(this)
			).fail(function () {
				this.flag.val(0);
				this.form.task.value = this.current_task;
				this.removeOverlay();
			}.bind(this));
		},

		checkScroll: function () {
			if ($(this.form).position().bottom < $(window).scrollTop()) {
				window.scroll(0, 0);
			}
		},

		checkTooltips: function () {
			// Remove MooTools tip
			$('.tip-wrap').remove();

			// Remove Bootstrap tip
			$('div.tooltip.fade.top').remove();

			// MooTools tooltips, todo - deprecated
			if (window.Tips) {
				var wrapper_id = this.wrapper.attr('id');

				document.id(wrapper_id).getElements('.hasTip').each(function (el) {
					var title = el.get('title');
					if (title) {
						var parts = title.split('::', 2);
						el.store('tip:title', parts[0]);
						el.store('tip:text', parts[1]);
					}
				});
				new Tips(document.id(wrapper_id).getElements('.hasTip'), {maxTitleChars: 50, fixed: false});
			}

			// Bootstrap tooltips
			if (jQuery.fn.tooltip) {
				$('#' + this.wrapper.attr('id') + ' .hasTooltip, [rel=tooltip]').tooltip({
					'html': true,
					'container': 'body',
				});
			}
		},

		checkPopovers: function () {
			if (jQuery.fn.popover) {
				// Remove Bootstrap popovers
				$('div.popover.fade.top').remove();

				var options = {
					'animation': true,
					'container': 'body',
					'delay': {
						'show': 50,
						'hide': 200
					},
					'html': true,
					'trigger': 'hover focus',
					'offset': [
						0,
						10
					],
					'boundary': 'scrollParent'
				};
				if (!this.Joomla4) {
					options = {
						'html': true,
						'trigger': 'hover focus',
						'container': 'body'
					};
				}

				// Bootstrap popovers
				$('#' + this.wrapper.attr('id') + ' .hasPopover, [rel=popover]').popover(options);
			}
		},

		checkContext: function () {
			$('#' + this.wrapper.attr('id') + ' .has-context')
				.mouseenter(function () {
					$('.btn-group', $(this)).show();
				})
				.mouseleave(function () {
					$('.btn-group', $(this)).hide();
					$('.btn-group', $(this)).removeClass('open');
				});
		},

		checkMultiselect: function () {
			if (!this.Joomla4) {
				return;
			}
			// Just disable actions by default.
			document.querySelectorAll('joomla-toolbar-button[list-selection]').forEach(el => {
				el.setDisabled(true);
				el.querySelectorAll('.dropdown-menu').forEach(el2 => {
					el2.classList.remove('show');
				});
			});

			// Disabled since we start working with 'multiselect' in checkScripts
			// // We can't emulate full behavior of native Joomla multiselect, just simple
			// // noinspection JSUnresolvedVariable
			// if (this.form.boxchecked) {
			// 	document.querySelectorAll('joomla-toolbar-button[list-selection]').forEach(el => {
			// 		// noinspection JSUnresolvedVariable
			// 		this.form.boxchecked.addEventListener('change', el.onChange);
			// 	});
			// 	this.form.querySelectorAll('input.form-check-input').forEach(el => {
			// 		el.addEventListener('change', () => {
			// 			let enabled = document.querySelectorAll('#' + this.form.id + ' tbody input.form-check-input:checked').length > 0;
			// 			document.querySelectorAll('joomla-toolbar-button[list-selection]').forEach(el => {
			// 				el.setDisabled(!enabled);
			// 			});
			// 		});
			// 	});
			// }
		},

		// Joomla4 list-view, table.columns etc.
		checkScripts: function () {
			if (document.querySelectorAll('.js-grid-item-action').length) {
				for (let s of document.scripts) {
					if (s.src.indexOf('/media/system/js/list-view.') !== -1) {
						this.rerunModule(s);
					} else if (s.src.indexOf('/media/system/js/table-columns.') !== -1) {
						// Apply column order only into our table.
						[...document.querySelectorAll('table:not(.columns-order-ignore)')].forEach(t => {
							if (t.parentNode.id !== 'ajaxtogglerWrapper') {
								t.classList.add('columns-order-ignore');
							}
						});
						// Remove existing button menu
						let dropdown = this.wrapper.prev('.dropdown.float-end.pb-2');
						if (dropdown.length) {
							dropdown.remove();
						}
						this.rerunModule(s);
					} else if (s.src.indexOf('/media/system/js/multiselect.') !== -1) {
						const old = document.addEventListener;
						document.addEventListener = (e, f) => {
							if (e === 'DOMContentLoaded') {
								f();
								document.addEventListener = old;
							}
						}
						this.rerunModule(s);
					}
				}
			}
		},

		/**
		 * @param {HTMLScriptElement} s
		 */
		rerunModule: function (s) {
			fetch(s.src).then(async r => {
				const ns = document.createElement('script');
				ns.type = 'module';
				ns.textContent = await r.text();
				document.body.prepend(ns);
			});
		},
	};
})(jQuery);
