<?php
/**
 * @package        AJAX Toggler
 * @copyright      Copyright (C)  2009 - 2022 AlterBrains.com. All rights reserved.
 * @license        GNU/GPL, see LICENSE.txt
 */
defined('_JEXEC') or die;

use Joomla\CMS\Environment\Browser;
use Joomla\CMS\Factory;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Response\JsonResponse;
use Joomla\CMS\Toolbar\Toolbar;
use Joomla\CMS\Uri\Uri;
use Joomla\String\StringHelper;

/** @noinspection PhpUnhandledExceptionInspection */
if (Factory::getUser()->id && Factory::getApplication()->isClient('administrator'))
{
	/**
	 * @since 1.0
	 */
	class plgSystemAjaxtoggler extends CMSPlugin
	{
		/**
		 * Application object.
		 *
		 * @var    \Joomla\CMS\Application\AdministratorApplication
		 * @since  3.5
		 */
		protected $app;

		/**
		 * @var bool
		 * @since 3.2.0
		 */
		protected $active;

		/**
		 * @var \Joomla\CMS\Document\HtmlDocument
		 * @since 3.2.3
		 */
		protected $document;

		/**
		 * @var \Joomla\CMS\Session\Session
		 * @since 3.2.3
		 */
		protected $session;

		public function onAfterRoute()
		{
			if (in_array($this->app->input->get('option'), $this->params->get('exclude', []), true))
			{
				return;
			}

			$this->session = $this->app->getSession();

			if ($this->app->input->getBool('jatoggler'))
			{
				$this->session->set('jatoggler', 1);

				// We should normal redirect in $app->redirect, else we fail with stupid IE
				$navigator = Browser::getInstance();
				if ($navigator->isBrowser('msie'))
				{
					$navigator->setBrowser('chrome');
				}
			}
		}

		public function onAfterDispatch()
		{
			/** @noinspection PhpUnhandledExceptionInspection */
			$this->document = version_compare(JVERSION, '4.0', '<') ? Factory::getDocument() : $this->app->getDocument();

			if ($this->document->getType() !== 'html' || !$this->session)
			{
				return;
			}

			$this->active = true;

			if ($this->session->get('jatoggler'))
			{
				$this->session->set('jatoggler', 0);

				if (strtolower($this->app->input->server->get('HTTP_X_REQUESTED_WITH', '')) === 'xmlhttprequest')
				{
					echo new JsonResponse([
						'html'       => $this->getTable() . $this->getOrdering(),
						'toolbar'    => $this->getToolbar(),
						'pagination' => $this->getPagination(),
					]);

					$this->app->close();
				}
			}
		}

		/**
		 * @since 3.3.0
		 */
		public function onBeforeCompileHead()
		{
			if (!$this->active)
			{
				return;
			}

			HTMLHelper::_('jquery.framework');

			$options = json_encode([
				'base' => Uri::root(),
			]);

			/** @noinspection BadExpressionStatementJS */
			$initScript = <<<JS
jQuery(function() {
	AjaxToggler.initialize({$options})
});
JS;

			$script = $this->_name . (JDEBUG ? '.js' : '.min.js');

			if (version_compare(JVERSION, '4.0', '<'))
			{
				/** @noinspection PhpDeprecationInspection */
				$this->document->addScript(Uri::root(true) . '/plugins/system/' . $this->_name . '/' . $script, ['version' => 'auto'])
					->addScriptDeclaration($initScript);
			}
			else
			{
				$this->document->getWebAssetManager()
					->registerAndUseScript('plg_' . $this->_name . '.admin', 'plugins/system/' . $this->_name . '/' . $script, ['version' => 'auto'], ['defer' => false], ['core'])
					->addInlineScript($initScript)
					->useScript('webcomponent.core-loader');
			}
		}

		/**
		 * @return string
		 * @since 1.0
		 */
		protected function getTable()
		{
			$html = '';

			// We have table
            // Apply greedy match to capture the full table
			if (preg_match_all('/(<table.*<\/table>)/s', $this->document->getBuffer('component'), $matches) && !empty($matches[1]))
			{
				foreach ($matches[1] as $table)
				{
					if (
						// Usual Joomla table
						StringHelper::strpos($table, ' class="table')
						// K2
						||
						StringHelper::strpos($table, ' class="adminlist table table-striped')
					)
					{
						$html = $table;
						break;
					}
				}
			}
			// No results! Keep our wrapper ID!
			else
			{
				$html = '<div id="ajaxtogglerWrapper" class="alert alert-no-items">' . Text::_('JGLOBAL_NO_MATCHING_RESULTS') . '</div>';
			}

			// small house-keeping
			return $html;
		}

		/**
		 * @return string
		 * @since 1.0
		 */
		protected function getToolbar()
		{
			$html = '';

			// Reload toolbar if Trashed state changed
			$state1 = $this->session->get('jatoggler_filter_published');
			$state2 = $this->app->input->get('filter_published', $this->app->input->get('filter_state'));

			// New Joomla 3.2.1+ filter
			$filter = $this->app->input->get('filter', [], 'array');

			if (isset($filter['published']))
			{
				$state2 = $filter['published'];
			}
			elseif (isset($filter['state']))
			{
				$state2 = $filter['state'];
			}

			if (
				// Reload toolbar if Trashed state changed
				(($state1 == -2 || $state2 == -2) && $state1 != $state2)
				||
				// Reload toolbar for selected components;
				($this->params->get('toolbar') && in_array($this->app->input->get('option'), $this->params->get('toolbar', []), true))
			)
			{
				$this->session->set('jatoggler_filter_published', $state2);

				// render toolbar
				$html = Toolbar::getInstance('toolbar')->render();
			}

			return $html;
		}

		/**
		 * @return string
		 * @since 1.0
		 */
		protected function getPagination()
		{
			$html = '';

			if (version_compare(JVERSION, '4.0', '<'))
			{
				$pattern = '/(<ul class="pagination-list">.*<\/ul>)/Us';
			}
			else
			{
				$pattern = '/(<ul class="pagination ms-auto mb-4 me-0">.*<\/ul>)/Us';
			}

			if (preg_match_all($pattern, $this->document->getBuffer('component'), $matches) && !empty($matches[0][0]))
			{
				$html = $matches[0][0] . '<input type="hidden" value="' . $this->app->input->getInt('limitstart') . '" name="limitstart" />';
			}

			return $html;
		}

		/**
		 * Re-init ordering list if table was ordered by ordering.
		 *
		 * @return string
		 * @since 1.0
		 */
		protected function getOrdering()
		{
			// Already loaded
			$exists = $this->app->input->get('jatoggler_ordering');

			$html = '';

			/** @noinspection PhpUnhandledExceptionInspection */
			$ordering = $this->app->input->get('filter_order');

			// New Joomla 3.2.1+ list
			$list = $this->app->input->get('list', [], 'array');

			if (isset($list['fullordering']))
			{
				$ordering = strtr($list['fullordering'], [
					' ASC'  => '',
					' DESC' => '',
				]);
			}

			if (in_array($ordering, ['a.ordering', 'ordering', 'a.lft', 'lft'], true))
			{
				if (version_compare(JVERSION, '4.0', '<'))
				{
					if (!empty($this->document->_script['text/javascript']))
					{
						preg_match_all('/var sortableList = ([^;]*);/', $this->document->_script['text/javascript'], $matches);

						if (!empty($matches[0][0]))
						{
							if (!$exists)
							{
								$mediaVersion = $this->document->getMediaVersion();

								$html .=
									'<script src="' . Uri::root(true) . '/media/jui/js/jquery.ui.core.min.js?' . $mediaVersion . '"></script>' .
									'<script src="' . Uri::root(true) . '/media/jui/js/jquery.ui.sortable.min.js?' . $mediaVersion . '"></script>' .
									'<script src="' . Uri::root(true) . '/media/jui/js/sortablelist.js?' . $mediaVersion . '"></script>' .
									'<link href="' . Uri::root(true) . '/media/jui/css/sortablelist.css?' . $mediaVersion . '" rel="stylesheet" />';
							}

							$html .= '<!--suppress JSUnusedLocalSymbols --><script>(function($){' . $matches[0][0] . '})(jQuery);</script>';
						}
					}
				}
				else
				{
					if (!$exists)
					{
						$html .= $this->extractScript('dragula');
						$html .= $this->extractStylesheet('dragula');
					}

					// Maximum perversion.
					//$html .= $this->extractScript('joomla.draggable');
					if (
						($item = $this->document->getWebAssetManager()->getAsset('script', 'joomla.draggable'))
						&& ($file = JPATH_SITE . '/' . $item->getUri())
						&& file_exists($file)
					)
					{
						$time = time();

						// Wrap into function to prevent "Uncaught SyntaxError: Identifier 'url' has already been declared"
						$html .= '<script>'
							. strtr('(function () {' . file_get_contents($file) . '}());', ['DOMContentLoaded' => 'DOMContentLoaded' . $time])
							. '; document.dispatchEvent(new CustomEvent("DOMContentLoaded' . $time . '", {}));'
							. '</script>';
					}

					// todo, allow custom options. Joomla doesn't add them by default.
					//$opts = $this->document->getScriptOptions('draggable-list');
				}
			}

			return $html;
		}

		/**
		 * @param string $name
		 *
		 * @return string|void
		 *
		 * @since 3.3.0
		 */
		protected function extractScript($name)
		{
			if ($item = $this->document->getWebAssetManager()->getAsset('script', $name))
			{
				$uri = $item->getUri();
				if ($version = $item->getVersion())
				{
					$uri .= (strpos('?', $uri) == false ? '?' : '&amp;') . $version;
				}

				return '<script src="' . $uri . '"></script>';
			}
		}

		/**
		 * @param string $name
		 *
		 * @return string|void
		 *
		 * @since 3.3.0
		 */
		protected function extractStylesheet($name)
		{
			if ($item = $this->document->getWebAssetManager()->getAsset('style', $name))
			{
				$uri = $item->getUri();
				if ($version = $item->getVersion())
				{
					$uri .= (strpos('?', $uri) == false ? '?' : '&amp;') . $version;
				}

				return '<link href="' . $uri . '" rel="stylesheet" />';
			}
		}
	}
}
