<?php
/**
 * @package        Alter Fields
 * @copyright      Copyright (C) 2022-2022 AlterBrains.com. All rights reserved.
 * @license        http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
 */

/** @noinspection PhpMultipleClassDeclarationsInspection */
defined('_JEXEC') or die();

use Joomla\CMS\Application\AdministratorApplication;
use Joomla\CMS\Document\HtmlDocument;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Profiler\Profiler;
use Joomla\Registry\Registry;
use Joomla\CMS\Uri\Uri;
use Joomla\Component\Fields\Administrator\Helper\FieldsHelper;
use Joomla\Event\Event;
use Joomla\Event\SubscriberInterface;

//use Joomla\Plugin\System\Alterfields\ListModelTrait;

if (Factory::getApplication()->isClient('administrator')) {
    // Joomla 3 legacy, todo remove
    if (version_compare(JVERSION, '4.0', '<')) {
        if (!interface_exists(SubscriberInterface::class, false)) {
            require_once __DIR__ . '/src/SubscriberInterface.php';
        }
        if (!class_exists(FieldsHelper::class)) {
            \JLoader::register('FieldsHelper', JPATH_ADMINISTRATOR . '/components/com_fields/helpers/fields.php');
            class_alias(\FieldsHelper::class, FieldsHelper::class);
        }
    }

    /**
     * @since        1.0
     * @noinspection PhpUnused
     *
     * @property AdministratorApplication $app Too much IDE issues with this weird native declaration.
     */
    class plgSystemAlterfields extends CMSPlugin implements SubscriberInterface
    {
        /**
         * @var string
         * @since 1.0
         */
        protected static $context;

        /**
         * @var array
         * @since 1.0
         */
        protected static $contextMap;

        /**
         * @var stdClass[]
         * @since 1.0
         */
        protected static $fields;

        /**
         * All fields, required to get subform children which can be missed in $fields.
         * @var stdClass[]
         * @since 1.2.0
         */
        protected static $fieldsAll;

        /**
         * @var Registry
         * @since 1.0
         */
        public static $listModelState;

        /**
         * @inheritDoc
         * @since 1.0
         */
        public function __construct(&$subject, $config = [])
        {
            parent::__construct($subject, $config);

            $this->app = Factory::getApplication();

            self::$contextMap = $this->params->get('contextMap');
        }

        /**
         * @inheritDoc
         * @since 1.0
         */
        public static function getSubscribedEvents(): array
        {
            return [
                'onContentPrepareForm' => 'onContentPrepareForm',
                'onAfterDispatch'      => 'onAfterDispatch',
            ];
        }

        /**
         * @paramJoomla4 Event $event Todo, remove Joomla3 compat
         * @param ...$args
         *
         * @since        1.0
         */
        public function onContentPrepareForm(/*Event $event*/ ...$args)
        {
            /** @var Form $form */
            /** @var \stdClass $data */
            [$form, $data] = $args[0] instanceof Event ? $args[0]->getArguments() : $args;

            switch ($form->getName()) {
                case 'com_fields.field.' . $this->app->input->get('context'):
                    if (!empty($data->type)) {
                        $this->loadLanguage();
                        $form->loadFile(__DIR__ . '/forms/com_fields.field.xml', false);
                        // todo, 'repeatable' is Joomla3 EOL
                        if ($data->type === 'subform' || $data->type === 'repeatable') {
                            $form->removeField('af_orderable', 'params');
                            if (!empty($data->params['af_orderable'])) {
                                $data->params['af_orderable'] = '0';
                            }
                        }
                    }
                    break;

                case 'com_fields.fields.filter':
                    if ($this->params->get('backend_fields_info')) {
                        $this->decorateFieldList();
                    }
                    break;
            }
        }

        /**
         * @since 1.0
         */
        public function onAfterDispatch()
        {
            JDEBUG and Profiler::getInstance('Application')->mark('before Alter Fields list');

            $this->decorateList();

            JDEBUG and Profiler::getInstance('Application')->mark('after Alter Fields list');
        }

        /**
         * @since 1.0
         */
        protected function decorateList()
        {
            // Get fields to display in columns
            $fields = static::getFields(static function ($field) {
                return $field->af_listable && empty($field->only_use_in_subform);
            });
            if (!$fields) {
                return;
            }

            /** @var HtmlDocument $document */
            $document = $this->app->getDocument();

            $document->setBuffer(
                preg_replace_callback(
                    '/<table(.+?)name="checkall-toggle"(.+?)<\/table>/s',
                    function ($matches) use ($fields) {
                        return $this->decorateListTable($matches, $fields);
                    },
                    $document->getBuffer('component'),
                    1
                ),
                'component'
            );
        }

        /**
         * @param   array       $matches
         * @param   stdClass[]  $fields
         *
         * @return string
         * @since 1.0
         */
        protected function decorateListTable(array $matches, array $fields)
        {
            // Extract thead.
            $matches[0] = preg_replace_callback(
                '/<thead(.+?)<\/thead>/us',
                function ($matches) use ($fields) {
                    return $this->decorateListTableHead($matches, $fields);
                },
                $matches[0],
                1
            );

            // Extract tbody.
            $matches[0] = preg_replace_callback(
                '/<tbody(.+?)<\/tbody>/us',
                function ($matches) use ($fields) {
                    return $this->decorateListTableBody($matches, $fields);
                },
                $matches[0],
                1
            );

            // Adjust footer for Joomla 3.
            if (version_compare(JVERSION, '4.0', '<')) {
                $matches[0] = preg_replace_callback('/<tfoot>(.*?)<td colspan="(\d+)">/us', static function ($matches2) use ($fields) {
                    return empty($matches2[2]) ? $matches2[0] : strtr($matches2[0], [
                        '="' . $matches2[2] . '"' => '="' . ((int)$matches2[2] + count($fields)) . '"',
                    ]);
                }, $matches[0], 1);
            }

            // Responsive table
            if (version_compare(JVERSION, '4.0', '<') && !\defined('ALTERBRAINS_TABLE_RESPONSIVE')) {
                \define('ALTERBRAINS_TABLE_RESPONSIVE', true);
                $matches[0] = '<div style="overflow-x: auto;">' . $matches[0] . '</div>';
            }

            return $matches[0];
        }

        /**
         * @param   array       $matches
         * @param   stdClass[]  $fields
         *
         * @return string
         * @since 1.0
         */
        protected function decorateListTableHead(array $matches, array $fields)
        {
            if (self::$listModelState) {
                $listOrder = htmlspecialchars(self::$listModelState->get('list.ordering'));
                $listDirn = htmlspecialchars(self::$listModelState->get('list.direction'));
            } else {
                $listOrder = '';
                $listDirn = '';
            }

            // Titles, note: can be <th> or <td>
            $titlesHtml = [];
            foreach ($fields as $field) {
                if ($field->af_orderable) {
                    // Field key in request.
                    $field->af_name = 'af_' . $field->id;

                    $title = HTMLHelper::_('searchtools.sort', $field->af_list_title, $field->af_name, $listDirn, $listOrder);
                } else {
                    $title = htmlspecialchars($field->af_list_title);
                }

                $class = $field->params->get('af_th_class');
                $titlesHtml[] = '<th' . ($class ? ' class="' . $class . '"' : '') . '>' . $title . ' </th>';
            }

            return strtr($matches[0], [
                '</tr>' => implode("\n", $titlesHtml) . '</tr>',
            ]);
        }

        /**
         * @param   array       $matches
         * @param   stdClass[]  $fields
         *
         * @return string
         * @since 1.0
         */
        protected function decorateListTableBody(array $matches, array $fields)
        {
            $tbody = $matches[0];

            // Values
            preg_match_all('/<td(.+?)name="cid\[]" value="(\d+)"(.+?)<\/tr>/us', $tbody, $matches);

            if (!isset($matches[0][0])) {
                return $tbody;
            }

            $replacements = [];

            $db = Factory::getDbo();

            $ids = array_map([$db, 'quote'], $matches[2]);

            $query = 'SELECT * 
				FROM #__fields_values 
				WHERE field_id IN(' . implode(',', array_keys($fields)) . ') AND item_id IN(' . implode(',', $ids) . ')';
            $db->setQuery($query);
            $allValues = $db->loadObjectList();

            $itemValues = [];

            foreach ($allValues as $value) {
                // Multiple? Transform to array.
                if (isset($itemValues[$value->item_id][$value->field_id])) {
                    if (!is_array($itemValues[$value->item_id][$value->field_id])) {
                        $itemValues[$value->item_id][$value->field_id] = [$itemValues[$value->item_id][$value->field_id]];
                    }

                    $itemValues[$value->item_id][$value->field_id][] = $value->value;

                    continue;
                }

                // String by default
                $itemValues[$value->item_id][$value->field_id] = $value->value;
            }

            PluginHelper::importPlugin('fields');

            foreach ($matches[0] as $i => $full) {
                $fieldsHtml = [];
                foreach ($fields as $field) {
                    $fieldHtml = '';

                    if (isset($itemValues[$matches[2][$i]][$field->id])) {
                        $field->value = $itemValues[$matches[2][$i]][$field->id];
                        $field->rawvalue = $field->value;

                        // Get rendered output, see FieldsHelper::getFields()

                        $item = new \stdClass();

                        // Event allow plugins to modify the output of the field before it is prepared
                        $this->app->triggerEvent('onCustomFieldsBeforePrepareField', [self::$context, $item, &$field]);

                        // Gathering the value for the field
                        $value = $this->app->triggerEvent('onCustomFieldsPrepareField', [self::$context, $item, &$field]);

                        if (is_array($value)) {
                            $value = implode(' ', $value);
                        }

                        // Event allow plugins to modify the output of the prepared field
                        $this->app->triggerEvent('onCustomFieldsAfterPrepareField', [self::$context, $item, $field, &$value]);

                        // Skip rendering empty values.
                        if ($value !== '' && $value !== []) {
                            // Assign the value
                            $field->value = $value;

                            // Some custom preparations
                            $this->prepareFieldBeforeListRender($field);

                            // Use plain output.
                            //$fieldHtml = $value;

                            // Note: frontend template paths are not used! Only backend template paths.
                            $fieldHtml = FieldsHelper::render(
                                self::$context, 'field.' . $field->params->get('af_layout', 'render'), [
                                    'item'    => $item,
                                    'context' => self::$context,
                                    'field'   => $field,
                                ]
                            );
                        }
                    }

                    $class = $field->params->get('af_td_class');
                    $style = $field->params->get('af_td_style');

                    $fieldHtml = '<td' . ($class ? ' class="' . $class . '"' : '') . ($style ? ' style="' . $style . '"' : '') . '>' . $fieldHtml . '</td>';

                    $fieldsHtml[] = $fieldHtml;
                }

                $replacements[$full] = strtr($full, [
                    '</tr>' => implode("\n", $fieldsHtml) . '</tr>',
                ]);
            }

            return strtr($tbody, $replacements);
        }

        /**
         * @param   stdClass  $field
         *
         * @since 1.0
         */
        protected function prepareFieldBeforeListRender($field)
        {
            /** @noinspection DegradedSwitchInspection */
            /** @noinspection PhpSwitchStatementWitSingleBranchInspection */
            switch ($field->type) {
                case 'media':
                    $field->value = strtr($field->value, [
                        ' src="' => ' style="max-width:100%;height:auto" src="' . Uri::root(true) . '/',
                    ]);
                    break;
            }
        }

        /**
         * @param   string  $realOption
         * @param   string  $realView
         * @param   string  $realModel  Can be case-sensitive, i.e. AdminEvents
         *
         * @return string|false
         * @since 1.0
         */
        public static function getContext(&$realOption = null, &$realView = null, &$realModel = null)
        {
            static $option, $view, $model;

            if (self::$context === null) {
                $app = Factory::getApplication();

                $map = static::getContextMap();

                $option = $app->input->get('option');

                // Note default empty string required for empty view
                $view = $app->input->get('view', '');

                // Special case for com_categories
                if ($option === 'com_categories') {
                    $view = 'categories';
                    $extension = $app->input->get('extension', $map['com_categories']['_default'] ?? 'com_content');

                    self::$context = $map[$option][$extension] ?? false;
                } else {
                    self::$context = $map[$option][$view] ?? false;

                    // Default view
                    if ($view === '' && isset($map[$option][''])) {
                        $view = $map[$option]['_default'];
                    }
                }

                $model = $map[$option]['_models'][$view] ?? $view;
            }

            $realOption = $option;
            $realView = $view;
            $realModel = $model;

            return self::$context;
        }

        /**
         * @return array
         * @since 1.0
         */
        protected static function getContextMap()
        {
            if (!is_array(static::$contextMap)) {
                $contextMap = explode(
                    "\n",
                    strtr(trim(static::$contextMap), [
                        "\r\n" => "\n",
                        ' '    => '',
                    ])
                );

                static::$contextMap = [];

                // Prepare map, each line is "fieldsContext|com_option:view1[?],[$view2]"
                foreach ($contextMap as $map1) {
                    if (('' === $map1 = trim($map1)) || strpos($map1, '|') === false) {
                        continue;
                    }

                    [$context, $componentView] = @explode('|', $map1, 2);

                    if (strpos($componentView, ':') !== false) {
                        [$component, $views] = explode(':', $componentView, 2);

                        foreach (explode(',', $views) as $view) {
                            // Custom model name for view
                            if (strpos($view, '>') !== false) {
                                [$view, $model] = explode('>', $view, 2);

                                static::$contextMap[$component]['_models'][$view] = $model;
                            }

                            // Default view
                            if (substr($view, -1) === '?') {
                                $view = substr($view, 0, -1);

                                static::$contextMap[$component][''] = $context;
                                static::$contextMap[$component]['_default'] = $view;
                            }

                            static::$contextMap[$component][$view] = $context;
                        }
                    }
                }
            }

            return static::$contextMap;
        }

        /**
         * @param   callable  $filterCallback
         *
         * @return stdClass[]
         * @since 1.0
         */
        public static function &getFields($filterCallback = null)
        {
            // Load all fields.
            if (self::$fieldsAll === null) {
                self::$fieldsAll = [];

                // Get all custom field instances
                if (false !== $context = static::getContext()) {
                    foreach (FieldsHelper::getFields($context, null, false, null, true) as $customField) {
                        static::$fieldsAll[$customField->id] = $customField;
                    }
                }
            }

            if (self::$fields === null) {
                self::$fields = [];

                if (false !== $context = static::getContext($option, $view)) {
                    // Detect item category
                    if ($filterCategoryId = Factory::getApplication()->getUserState($option . '.' . $view . '.filter.category_id')) {
                        $item = (object)['fieldscatid' => $filterCategoryId];
                    }

                    // All fields, including subform-only fields.
                    $fields = [];
                    foreach(FieldsHelper::getFields($context, $item ?? null, false, [], true) as $field) {
                        $fields[$field->id] = $field;
                    }

                    // Subform-only fields can be missed if we have selected category, inject from allFields cache.
                    foreach ($fields as $field) {
                        // Subform contains children in options.
                        if ($field->type === 'subform') {
                            foreach ($field->fieldparams->get('options', []) as $opt) {
                                if (!isset($fields[$opt->customfield])) {
                                    if (!isset(self::$fieldsAll[$opt->customfield])) {
                                        continue;
                                    }
                                    $fields[$opt->customfield] = self::$fieldsAll[$opt->customfield];
                                }
                            }
                        }
                    }

                    foreach ($fields as $field) {
                        self::$fields[$field->id] = $field;

                        // Hide display label
                        $field->params->set('showlabel', 0);

                        $field->af_list_title = Text::_(
                            $field->af_list_title ?? $field->params->get('af_list_title', $field->title)
                        );

                        $field->af_listable = $field->params->get('af_listable');
                        $field->af_orderable = $field->params->get('af_orderable');
                        $field->af_searchable = $field->params->get('af_searchable');
                        $field->af_filterable = $field->params->get('af_filterable');

                        // subform and children are not orderable. todo - repeatable is Joomla3 EOL
                        if (!empty($field->only_use_in_subform) || $field->type === 'subform' || $field->type === 'repeatable') {
                            $field->af_orderable = false;
                        }
                    }
                }
            }

            if (!$filterCallback) {
                return self::$fields;
            }

            $fields = [];
            foreach (self::$fields as $field) {
                if ($filterCallback($field)) {
                    $fields[$field->id] = $field;
                }
            }

            return $fields;
        }

        /**
         * @since 1.2.0
         */
        protected function decorateFieldList()
        {
            $debug = debug_backtrace(DEBUG_BACKTRACE_PROVIDE_OBJECT, 4);
            $classname = version_compare(JVERSION, '4.0', '<') ? \FieldsModelFields::class : \Joomla\Component\Fields\Administrator\Model\FieldsModel::class;
            if (!empty($debug[3]['object']) && $debug[3]['object'] instanceof $classname) {
                foreach ($debug[3]['object']->getItems() as $item) {
                    $badges = \array_filter([
                        $item->params->get('af_listable') ? Text::_('PLG_SYSTEM_ALTERFIELDS_BACKEND_FIELDS_INFO_LISTABLE') : '',
                        $item->params->get('af_orderable') ? Text::_('PLG_SYSTEM_ALTERFIELDS_BACKEND_FIELDS_INFO_ORDERABLE') : '',
                        $item->params->get('af_filterable') ? Text::sprintf('PLG_SYSTEM_ALTERFIELDS_BACKEND_FIELDS_INFO_FILTERABLE', $this->getSearchModeTitle($item->params->get('af_filter_mode'))) : '',
                        $item->params->get('af_searchable') ? Text::sprintf('PLG_SYSTEM_ALTERFIELDS_BACKEND_FIELDS_INFO_SEARCHABLE', $this->getSearchModeTitle($item->params->get('af_search_mode'))) : '',
                    ]);
                    if ($badges) {
                        $item->note = $item->note ? $item->note . '. ' : '';
                        $item->note .= implode(', ', $badges);
                    }
                }
            }
        }

        /**
         * @param   string  $mode
         *
         * @return  string
         *
         * @since 1.2.0
         */
        /**
         * @param   string  $mode
         *
         * @return  string
         *
         * @since 1.2.0
         */
        protected function getSearchModeTitle($mode)
        {
            return $mode ? Text::_('PLG_SYSTEM_ALTERFIELDS_FIND_MODE_' . $mode) : 'n/a';
        }
    }
}
